/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.scroller;

import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import org.gerhardb.lib.dirtree.filelist.FileListSelectionModel;

/**
 * Commands to scroll among a group of pages. Used for making a slide show with
 * sound. Class is a singleton.
 */
public class ScrollerKeyListener implements KeyListener
{
	private KeypadOps myKeypadOps = null;
	private StringBuffer myNumberJumpBuffer = new StringBuffer(12);
	private boolean iAmCollectingNumbers = false;
	private boolean iAmCollectingNegativeNumbers = false;	
	private FileListSelectionModel mySelectionModel;	
	private Scroller myScroller;
	private boolean iAmSelecting = false;	
	private boolean iAmDeselecting = false;
	private boolean iMadeASelection = false;
	private int anchorIndexForCtrlS = -1;
	
	public ScrollerKeyListener(Scroller scroller)
	{
		this.myScroller = scroller;
	}
	
	public void setKeypadOps(KeypadOps kpo)
	{
		this.myKeypadOps = kpo;
	}
	
	public void setListSelectionModel(FileListSelectionModel sm)
	{
		this.mySelectionModel = sm;	
	}

	// =========================================================================
	// KeyListener Implementation
	// =========================================================================
	public void keyPressed(KeyEvent event)
	{
		if (event.isConsumed()) { return; }		
		if (this.myScroller.myShowViewKeyListener != null)
		{
			this.myScroller.myShowViewKeyListener.keyReleased(event);
			if (event.isConsumed())
			{
				return;
			}
		}		
		//System.out.println("ScrollerKeyListener keyPressed: event.getKeyCode(): " + event.getKeyCode() + "  KEY: " + KeyEvent.getKeyText(event.getKeyCode()));
		
		switch (event.getKeyCode())
		{
		case KeyEvent.VK_CONTROL:
			this.iAmCollectingNumbers = true;
			return;
		case KeyEvent.VK_ALT:
			this.iAmCollectingNumbers = true;
			this.iAmCollectingNegativeNumbers = true;
			return;
		//case KeyEvent.VK_SPACE:
		//	spaceIsDown = true;
		//	return;		
		case KeyEvent.VK_S: // keyPressed
			if (!event.isControlDown() && !event.isAltDown() && !event.isMetaDown())
			{
				this.iAmSelecting = true;
				this.iAmDeselecting = false;
				this.iMadeASelection = false;
				this.anchorIndexForCtrlS = this.myScroller.getValueZeroBased();
				break;
			}
			return;
		case KeyEvent.VK_D: // keyPressed
			if (!event.isControlDown() && !event.isAltDown() && !event.isMetaDown())
			{
				this.iAmDeselecting = true;
				this.iAmSelecting = false;
				this.iMadeASelection = false;
				this.anchorIndexForCtrlS = this.myScroller.getValueZeroBased();
				break;
			}
			return;
		}
	}

	public void keyReleased(KeyEvent event)
	{
		//System.out.println("BEFORE ScrollerKeyListener keyReleased: event.getKeyCode(): " + event.getKeyCode() + "  KEY: " + KeyEvent.getKeyText(event.getKeyCode()));
		if (event.isConsumed()) { return; }
		if (this.myScroller.myShowViewKeyListener != null)
		{
			this.myScroller.myShowViewKeyListener.keyReleased(event);
			if (event.isConsumed())
			{
				return;
			}
		}
		//System.out.println("AFTER ScrollerKeyListener keyReleased: event.getKeyCode(): " + event.getKeyCode() + "  KEY: " + KeyEvent.getKeyText(event.getKeyCode()));

		// Used for jump to numbers entered with alt or ctrl.
		if (this.iAmCollectingNumbers)
		{
			if (this.iAmCollectingNegativeNumbers)
			{
				switch (event.getKeyCode())
				{
				case KeyEvent.VK_MINUS: // Regular minus
					if (this.myNumberJumpBuffer.length() == 0)
					{
						// This is how you enter a negative number.
						this.myNumberJumpBuffer.append("-"); //$NON-NLS-1$
					}
					break;
				case 109: // Minus on the keyapd
					if (this.myNumberJumpBuffer.length() == 0)
					{
						// This is how you enter a negative number.
						this.myNumberJumpBuffer.append("-"); //$NON-NLS-1$
					}
					break;
				}
			}
			switch (event.getKeyCode())
			{
			case KeyEvent.VK_NUMPAD0:
			case KeyEvent.VK_NUMPAD1:
			case KeyEvent.VK_NUMPAD2:
			case KeyEvent.VK_NUMPAD3:
			case KeyEvent.VK_NUMPAD4:
			case KeyEvent.VK_NUMPAD5:
			case KeyEvent.VK_NUMPAD6:
			case KeyEvent.VK_NUMPAD7:
			case KeyEvent.VK_NUMPAD8:
			case KeyEvent.VK_NUMPAD9:
				this.myNumberJumpBuffer.append(KeyEvent
						.getKeyText(event.getKeyCode() - 48));
				return;
			case KeyEvent.VK_0:
			case KeyEvent.VK_1:
			case KeyEvent.VK_2:
			case KeyEvent.VK_3:
			case KeyEvent.VK_4:
			case KeyEvent.VK_5:
			case KeyEvent.VK_6:
			case KeyEvent.VK_7:
			case KeyEvent.VK_8:
			case KeyEvent.VK_9:
				this.myNumberJumpBuffer.append(KeyEvent.getKeyText(event.getKeyCode()));
				return;
			}
		}

		switch (event.getKeyCode())
		{
		case KeyEvent.VK_PAGE_DOWN:
			this.myScroller.pageDown();
			updateSelectionInterval();
			break;
		case KeyEvent.VK_PAGE_UP:
			this.myScroller.pageUp();
			updateSelectionInterval();
			break;
		case KeyEvent.VK_HOME:
			this.myScroller.home();
			updateSelectionInterval();
			break;
		case KeyEvent.VK_END:
			this.myScroller.end();
			updateSelectionInterval();
			break;
		case KeyEvent.VK_UP:
		case KeyEvent.VK_LEFT:
			this.myScroller.up();
			updateSelectionInterval();
			break;
		case KeyEvent.VK_DOWN:
		case KeyEvent.VK_RIGHT:
			this.myScroller.down();
			updateSelectionInterval();
			break;
			/*
			 * Switched to Insert being repeat so shift space no longer needed
			 * for repeate.
		case KeyEvent.VK_SHIFT:
			//System.out.println("ScrollerKeyListner SHIFT");
			if (spaceIsDown)
			{
	 			myKeypadOps.clickRepeat();
	 			userIsRepeating = true;
			}
			else 
			{
	 			userIsRepeating = false;
			}
			break;
		case KeyEvent.VK_SPACE:
			//System.out.println("ScrollerKeyListner SPACE");
			spaceIsDown = false;
			if (event.isShiftDown())
			{
	 			myKeypadOps.clickRepeat();
	 			userIsRepeating = true;
			}
			else if (userIsRepeating)
			{
	 			userIsRepeating = false;
			}
			else
			{
				if (event.isControlDown())  
				{
					myScroller.up();
					updateSelectionInterval();
				}
				else
				{
					myScroller.down();
					updateSelectionInterval();
				}
			}
			break;
			*/

		case KeyEvent.VK_SPACE:
			//System.out.println("ScrollerKeyListner SPACE");
			if (event.isControlDown())  
			{
				this.myScroller.up();
				updateSelectionInterval();
			}
			else
			{
				this.myScroller.down();
				updateSelectionInterval();
			}
			break;
			
		case KeyEvent.VK_ENTER: // regular and keypad
			// the 429 problem is here!!
			/*
			if (!event.isControlDown() && !event.isAltDown() 
					&& !event.isShiftDown() 
					&& !event.isMetaDown()  && !event.isAltGraphDown())
			{
	 			myKeypadOps.clickRepeat();
			}
			*/
			break;
		case KeyEvent.VK_DELETE: // regular and keypad if num lock off
			// Done by menu accelerators unless full screen.
			if (this.myKeypadOps.isFullScreen())
			{
				this.myKeypadOps.clickTrash();
			}
			break;
		case KeyEvent.VK_INSERT: // regular and keypad if num lock off
			if (this.myKeypadOps.isFullScreen())
			{
				if (event.isControlDown())
				{
					this.myKeypadOps.clickPark();
				}
				else if (event.isShiftDown())
				{
				// Do Nothing. myKeypadOps.clickDirectoryList();
				}
				else if (event.isAltDown())
				{
					// Do Nothing.
				}
				else
				{
					this.myKeypadOps.clickRepeat();
				}
			}
			break;	
		case KeyEvent.VK_M: // for CTRL-M
			if (event.isControlDown())
			{
				this.myKeypadOps.clickDirectoryList();
			}
			break;
		case KeyEvent.VK_S: 
			if (this.iMadeASelection == false)
			{
				updateSelectionInterval();
			}
			this.iAmSelecting = false;
			break;
		case KeyEvent.VK_D: 
			if (this.iMadeASelection == false)
			{
				updateSelectionInterval();
			}
			this.iAmDeselecting = false;
			break;
			

		// NUMBER KEYPAD
		// NOTE THAT KEYPAD CAN ALSO BE USED TO JUMP TO IF ctrl OR alt USED
		// ctrl IS JUMP TO ABSOLUTE POSITION
		// alt IS JUMP TO RELETIVE POSITION USING + - FOR DIRECTION
		case 106: // NumPad *
			this.myScroller.jumpTo(this.myScroller.myIndex + 10);
			break;
		case 111: // NumPad /
			this.myScroller.jumpTo(this.myScroller.myIndex - 10);
			break;
		case 109: // NumPad -
			if (!event.isAltDown())
			{
				this.myScroller.up();
			}
			break;
		case 107: // NumPad +
			if (!event.isAltDown())
			{
				this.myScroller.down();
			}
			break;
		case KeyEvent.VK_NUMPAD3: // File down
			if (!event.isAltDown() && !event.isControlDown())
			{
				this.myScroller.pageDown();
			}
			break;
		case KeyEvent.VK_NUMPAD9: // File up
			if (!event.isAltDown() && !event.isControlDown())
			{
				this.myScroller.pageUp();
			}
			break;
		case KeyEvent.VK_NUMPAD7: // home
			if (!event.isAltDown() && !event.isControlDown())
			{
				this.myScroller.home();
			}
			break;
		case KeyEvent.VK_NUMPAD1: // end
			if (!event.isAltDown() && !event.isControlDown())
			{
				this.myScroller.end();
			}
			break;

		// If num lock is ON, either use number or scroll.
		// Don't move picture.
		case KeyEvent.VK_NUMPAD8: // up
		case KeyEvent.VK_NUMPAD4: // left
			if (!event.isAltDown() && !event.isControlDown())
			{
				this.myScroller.up();
			}
			break;
		case KeyEvent.VK_NUMPAD2: // down
		case KeyEvent.VK_NUMPAD6: // right
			if (!event.isAltDown() && !event.isControlDown())
			{
				this.myScroller.down();
			}
			break;

		// If num lock is OFF, either move picture or scroll.
		case KeyEvent.VK_KP_UP: // up
		case KeyEvent.VK_KP_LEFT: // left
			this.myScroller.up();
			break;
		case KeyEvent.VK_KP_DOWN: // down
		case KeyEvent.VK_KP_RIGHT: // right
			this.myScroller.down();
			break;
		case KeyEvent.VK_NUMPAD0: // insert
			if (!event.isAltDown() && !event.isControlDown())
			{
				// needed if numlock is on.  
				// Otherwise jibs.properties acclerator controls.
				this.myKeypadOps.clickPark();
			}
			break;
		// period on the numeric keypad only...
		case 110: // delete on the numeric keypad
			if (!event.isAltDown() && !event.isControlDown())
			{
				// needed if numlock is on.
				// Otherwise jibs.properties acclerator controls.
				this.myKeypadOps.clickTrash();
			}
			break;

		// TIME TO READ THE NUMBER PAD FOR A NUMERIC JUMP
		case KeyEvent.VK_CONTROL:
		{
			if (this.myNumberJumpBuffer.length() > 0)
			{
				String number = this.myNumberJumpBuffer.toString();
				// System.out.println("CONTROL RELEASED: " + number);
				try
				{
					int jumpTo = Integer.parseInt(number);
					this.myScroller.jumpTo(jumpTo - 1);
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
				this.myNumberJumpBuffer = new StringBuffer(12);
			}
			this.iAmCollectingNumbers = false;
			break;
		}
		case KeyEvent.VK_ALT:
		{
			if (this.myNumberJumpBuffer.length() > 0)
			{
				String number = this.myNumberJumpBuffer.toString();
				try
				{
					int jumpBy = Integer.parseInt(number);
					this.myScroller.jumpTo(this.myScroller.myIndex + jumpBy);
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
				this.myNumberJumpBuffer = new StringBuffer(12);
			}
			this.iAmCollectingNumbers = false;
			this.iAmCollectingNegativeNumbers = false;
			break;
		}
		default:
		}
		event.consume();
	}

	public void keyTyped(KeyEvent event)
	{
		// We don't care
	}

	// =========================================================================
	// Other
	// =========================================================================
	protected void updateSelectionInterval()
	{
		if (this.iAmSelecting)
		{
			this.iMadeASelection = true;					
			int curItem = this.myScroller.getValueZeroBased();
			this.mySelectionModel.addSelectionInterval(this.anchorIndexForCtrlS, curItem);	
		}
		else if (this.iAmDeselecting)
		{
			this.iMadeASelection = true;					
			int curItem = this.myScroller.getValueZeroBased();
			this.mySelectionModel.removeSelectionInterval(this.anchorIndexForCtrlS, curItem);	
		}
	}
	
	void clearSelection()
	{
		this.mySelectionModel.clearSelection();
	}
}
