/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.scroller;

import javax.swing.JSlider;

/**
 * Basic slider designed to work with the Scroller.
 */
public class ScrollerSlider extends JSlider implements ScrollerListener
{
	Scroller myScroller;
	
   //==========================================================================
   //                         Constructor
   //==========================================================================
   public ScrollerSlider(int newOrientation, Scroller scroller)
   {
      super(newOrientation, 0, 100, 0);
      this.myScroller = scroller;
      super.setModel(this.myScroller);
      setSlider();
      // Hook up to Global Scroller
      this.myScroller.addScrollerListener(this);
   }

   //==========================================================================
   //                         ScrollerListener Implementation
   //==========================================================================
   public void scrollerChanged(ScrollerChangeEvent ce)
   {
      // Make sure ticks always show.
      if (!this.myScroller.getValueIsAdjusting())
      {
      	setSlider();
      }
   }

   //==========================================================================
   //                         Private
   //==========================================================================
   /**
    * Adjust slider for a new list.
    */
   private void setSlider()
   {
      // setting mySlider value automatically generates update
      // Do first so other adjustments (setMax) don't change.
      super.setValue(this.myScroller.getValue());
      
      int count = getModel().getMaximum();
      // Handle when called before initialized.
      if (count < 0)
      {
         return;
      }
      
      super.setMinimum(0);      
      super.setMaximum(this.myScroller.getMaximum());
      
      // Set tick spacing
      int minorTick = 1;
      int majorTick = 1;
      if (count > 1 && count <= 10)
      {
         minorTick = 1;
         majorTick = 1;
      }
      else if (count > 10 && count <= 100)
      {
         minorTick = 1;
         majorTick = 5;
      }
      else if (count > 100 && count <= 300)
      {
         minorTick = 5;
         majorTick = 10;
      }
      else if (count > 300 && count <= 1000)
      {
         minorTick = 10;
         majorTick = 50;
      }
      else if (count > 1000)
      {
         minorTick = count / 100; // int for truncation
         majorTick = count / 20; // for truncation
      }

      //System.out.println( "Count: " + count + "  minor: "
      //+ minorTick + " Major: " + majorTick );
      super.setMinorTickSpacing(minorTick);
      super.setMajorTickSpacing(majorTick);

      if (minorTick == 1 || majorTick == 1)
      {
         super.setSnapToTicks(true);
      }
      else
      {
         super.setSnapToTicks(false);
      }

     super.setLabelTable(super.createStandardLabels(majorTick));

     // Set painting only after setting what the labels are!
     // AND after creating them!!!!
     // Especially when there were previously no labels.
     super.setPaintTicks(true);
     super.setPaintLabels(true);
     super.setEnabled(true);
   }
}
