/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.util;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.EventQueue;
import java.awt.KeyboardFocusManager;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyVetoException;
import java.beans.VetoableChangeListener;
import java.util.Date;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BoundedRangeModel;
import javax.swing.DefaultBoundedRangeModel;
import javax.swing.Icon;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JLayeredPane;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.KeyStroke;
import javax.swing.RootPaneContainer;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.WindowConstants;

import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.util.startup.ILoadingMessage;

/**
 * Blocks user interaction during a lengthy event during which the application
 * needs to behave modally providing a modal dialog box with a progress meter
 * if the task is lengthy.  All done out of the AWT thread so the application
 * can be providing visual feedback to the user as the lengthy operation is
 * processed.
 *
 * Based on WaitDialog and DialogModality developed by Craig Pell.
 */
public class ModalProgressDialogFlex implements ILoadingMessage
{
	// milliseconds To Wait before showing.
	final private String FINISHED_SYNC = "sync"; //$NON-NLS-1$
	final static long DEFAULT_THRESHOLD = 2000;
	long myThreshold = DEFAULT_THRESHOLD;
	Window myTopWindow;
	public static final Icon WAIT_ICON = Icons.getIcon(Icons.HOUR_GLASS);
	
	BoundedRangeModel myRange;
	Thread myTaskThread;
	JDialog myDialog;
	private JOptionPane myOptionPane;
	KeyboardFocusManager myKeyboardFocusManager;
   private int myIncrement = 1;
	
	private Date myShowTime;
	boolean iAmFinished = false;

	/*
   public static void popUp(
         Window topWindow,
         String titleBar,
         String explanation,
         String cancelBtnText,
         BoundedRangeModel range,
         Runnable runMe
         )
      {
      	popUp(
      	      topWindow,
      	      titleBar,
      	      explanation,
      	      cancelBtnText,
      	      range,
      	      runMe, 
      	      DEFAULT_THRESHOLD
      	      );
      }
      */
	
  /**
    * Dialog is guaranteed to come up by threshold even if the first event
    * has not processed.  When the first event finishes, it is used to estimate
    * the time to complete.  If that is greater than the threshold, the dialog
    * comes up immediately.  If not, it will be reevaluated each event.  If
    * all events actually finish before the threshold comes up, the dialog
    * never appears.  There is still a small chance the box could come up and
    * go away very fast, but it is unlikely.  Also, avoiding that entirely
    * would require guessing on the speed of the last few events and maybe
    * those are where it slows down.
    *
    * @param parent Component
    * @param titleBar String
    * @param explanation String
    * @param myCancelBtnText String
    * @param threshold long
    * @param range BoundedRangeModel if null, then indeterminate
    * @param runMe Runnable
    * @param milli How many seconds to wait.  Zero for immediate.
    */
   public static void popUp(
      Window topWindow,
      String titleBar,
      String explanation,
      String cancelBtnText,
      BoundedRangeModel range,
      Runnable runMe,
      long thresholdMilliSeconds
      )
   {
      if (topWindow == null)
      {
         System.out.println("ModalProgressDialog got null topWindow"); //$NON-NLS-1$
         return;
      }
 		ModalProgressDialogFlex flex = new ModalProgressDialogFlex(topWindow,
	         titleBar, explanation, cancelBtnText, range, thresholdMilliSeconds);
		flex.run(runMe);
  }
	
	/**
	 * Dialog is guaranteed to come up by threshold even if the first event
	 * has not processed.  When the first event finishes, it is used to estimate
	 * the time to complete.  If that is greater than the threshold, the dialog
	 * comes up immediately.  If not, it will be reevaluated each event.  If
	 * all events actually finish before the threshold comes up, the dialog
	 * never appears.  There is still a small chance the box could come up and
	 * go away very fast, but it is unlikely.  Also, avoiding that entirely
	 * would require guessing on the speed of the last few events and maybe
	 * those are where it slows down.
	 *
	 * @param parent Component
	 * @param titleBar String
	 * @param explanation String
	 * @param myCancelBtnText String
	 * @param threshold long
	 * @param range BoundedRangeModel if null, then indeterminate
	 * @param milli How many seconds to wait.  Zero for immediate.
	 */
	public ModalProgressDialogFlex(Window topWindow, String title,
			String message, String cancelBtnText, BoundedRangeModel range,
	      long thresholdMilliSeconds)
	{
		if (topWindow == null)
		{
			System.out.println("ModalProgressDialog got null topWindow"); //$NON-NLS-1$
			return;
		}
		this.myTopWindow = topWindow;
		this.myRange = range;
		this.myThreshold = thresholdMilliSeconds;
		JProgressBar progressBar = null;
		if (this.myRange == null)
		{
			this.myRange = new DefaultBoundedRangeModel();
			progressBar = new JProgressBar(this.myRange);
			progressBar.setIndeterminate(true);
		}
		else
		{
			progressBar = new JProgressBar(this.myRange);
		}
		Action cancelAction = new AbstractAction(UIManager
				.getString("OptionPane.cancelButtonText")) //$NON-NLS-1$
		{
			public void actionPerformed(ActionEvent event)
			{
				//System.out.println("CANCEL BUTTON PRESSED");
				interrupt();
			}
		};

		// Set up viewable panel
		JPanel option = new JPanel(new BorderLayout());
		option.add(progressBar, BorderLayout.NORTH);

		// Set up cancel button
		if (cancelBtnText != null)
		{
			JButton cancelButton = new JButton(cancelAction);
			cancelButton.setText(cancelBtnText);
			cancelButton.setDefaultCapable(true);
	
			// Default keystroke actions for cancel button
			Object actionID = "cancel"; //$NON-NLS-1$
			cancelButton.getActionMap().put(actionID, cancelAction);
			InputMap inputMap = cancelButton
					.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
			inputMap.put(KeyStroke.getKeyStroke("ESCAPE"), actionID); //$NON-NLS-1$
			inputMap.put(KeyStroke.getKeyStroke("CANCEL"), actionID); //$NON-NLS-1$
			inputMap.put(KeyStroke.getKeyStroke("STOP"), actionID); //$NON-NLS-1$
			option.add(cancelButton, BorderLayout.SOUTH);
			this.myOptionPane = new JOptionPane(
					" ", //$NON-NLS-1$
					JOptionPane.PLAIN_MESSAGE, JOptionPane.DEFAULT_OPTION, WAIT_ICON,
					new Object[] { option }, cancelButton);
		}
		else
		{
			this.myOptionPane = new JOptionPane(
					" ", //$NON-NLS-1$
					JOptionPane.PLAIN_MESSAGE, JOptionPane.DEFAULT_OPTION, WAIT_ICON,
					new Object[] { option });
		}

		this.myOptionPane.setMessage(message);
		this.myDialog = this.myOptionPane.createDialog(this.myTopWindow, title);

		this.myDialog.setModal(false);
		this.myDialog.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

		// If the owning windows goes away, we want to also.
		// Also, if user closes the myDialog stop everything.
		this.myDialog.addHierarchyListener(new HierarchyListener()
		{
			public void hierarchyChanged(HierarchyEvent event)
			{
				long flags = event.getChangeFlags();
				Component component = event.getComponent();
				if ((flags & HierarchyEvent.SHOWING_CHANGED) != 0)
				{
					if (!component.isShowing())
					{
						//System.out.println("hierarchyChanged");
						interrupt();
					}
				}
			}
		});
	}

	public void run(Runnable runMe)
	{
		this.myTaskThread = new Thread(runMe);
		Thread dialogThread = null;

		// This is the timer thread to see if we should show the dialog/
		// Based strictly on time in case the first event does not occur
		// before this time runs out.
		dialogThread = new Thread()
		{
			@Override
			public void run()
			{
				try
				{
					if (ModalProgressDialogFlex.this.myThreshold != 0)
					{
						Thread.sleep(ModalProgressDialogFlex.this.myThreshold);
					}
					if (ModalProgressDialogFlex.this.myTaskThread.isAlive())
					{
						EventQueue.invokeLater(new Runnable()
						{
							public void run()
							{
								activate();
							}
						});
					}
				}
				catch (InterruptedException e)
				{
					interrupt();
				}
			}
		};

		// The join will cause this thread to block until the joined thread
		// finishes.  Once joined, the cleanup routines are called.
		// This way we clean everything up when the task thread is done.
		Thread waitThread = new Thread()
		{
			@Override
			public void run()
			{
				try
				{
					ModalProgressDialogFlex.this.myTaskThread.join();
				}
				catch (InterruptedException e)
				{
					// fall through
				}

				synchronized (ModalProgressDialogFlex.this.FINISHED_SYNC)
				{
					ModalProgressDialogFlex.this.iAmFinished = true;
				}

				// april 8 - took out event queue from around uninstall
				// and deactivate.  Not sure why anyone ever thought
				// it would be needed.  It kept window from going away
				// while there is a long, unrelated refresh going on
				// behind this.

				// uninstall undoes the install which is ALWAYS done
				// just below this routine.
				// Call before deactivate() just in case deactivate has
				// a problem.
				try
				{
					uninstall();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}

				// deactivate undoes the dialog if it is ever activated.
				deactivate();
			}
		};

		// This is where the blocking starts.
		install();

		// *** START EVERYTHING UP ***
		// Starts the actual work!
		this.myTaskThread.start();

		// Starts the thread to see if we should display the dialog.
		dialogThread.start();

		// Waits for the actual work to finish.
		// waitThread must be started after the myTaskThread or else it has no
		// thread to join.
		waitThread.start();
	}

	// =========================================================================
	// Public Methods
	// =========================================================================

	/**
	 * Halts any executing task, and hides this object's dialog if it
	 * is displayed.  The executing task, if any, will be interrupted;
	 * it is up to that task to terminate itself based on its
	 * interruption status.
	 */
	public void interrupt()
	{
		if (this.myTaskThread != null)
		{
			this.myTaskThread.interrupt();
		}
	}

   public void setText(String msg)
	{
		this.myOptionPane.setMessage(msg);
	}
   
	public void setMessage(String msg)
	{
		this.myOptionPane.setMessage(msg);
	}

   public int getNextIncrement(){return this.myIncrement++;}

	// =========================================================================
	// Private Methods
	// =========================================================================

	synchronized void activate()
	{
		// Don't bother if the work is already done.
		synchronized (this.FINISHED_SYNC)
		{
			if (this.iAmFinished == true) { return; }
		}
		this.myDialog.setVisible(true);
		this.myShowTime = new Date();
	}

	synchronized void deactivate()
	{
		if (this.myShowTime == null) { return; }

		// Show for at least a second, even if done.
		// To avoid flicker.
		long time = new Date().getTime() - this.myShowTime.getTime();
		if (time < 1000)
		{
			try
			{
				Thread.sleep(1000);
				if (this.myTaskThread != null && ModalProgressDialogFlex.this.myTaskThread.isAlive())
				{
					EventQueue.invokeLater(new Runnable()
					{
						public void run()
						{
							activate();
						}
					});
				}
			}
			catch (InterruptedException e)
			{
				interrupt();
			}
		}

		//System.out.println("DEACTIVATE CALLED");
		this.myDialog.setVisible(false);
		this.myDialog.dispose();
		//System.out.println("DEACTIVATE COMPLETED");
	}

	// =========================================================================
	// Glass Pane Methods (from DialogModality)
	// =========================================================================
	/**
	 * Makes this object's dialog appear modal with
	 * respect to its owner window.
	 *
	 */
	private void install()
	{
		this.myTopWindow.requestFocus();

		//  Prevents window from getting focus.
		// This (windowFocusPreventer) is what prevents keystrokes
		// from working in the window!!!
		this.myKeyboardFocusManager = KeyboardFocusManager
				.getCurrentKeyboardFocusManager();
		//myKeyboardFocusManager.addVetoableChangeListener("focusedWindow", windowFocusPreventer);

		this.myTopWindow.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

		// Adjusts the size of the mouseBlocker if the user changes
		// the screen size.  mouseBlocker is a clear panel which eats
		// mouse clicks.
		this.myTopWindow.addComponentListener(this.mouseBlockerSynchronizer);

		if (this.myTopWindow instanceof RootPaneContainer)
		{
			JLayeredPane l = ((RootPaneContainer) this.myTopWindow).getLayeredPane();
			this.mouseBlocker.setSize(l.getSize());
			l.add(this.mouseBlocker, new Integer(Integer.MAX_VALUE));
		}
		else
		{
			// better than nothing...
			this.myTopWindow.setEnabled(false);
		}
		//System.out.println("INSTALL COMPLETED");
	}

	/**
	 * Removes all influences of this object over its dialog.
	 */
	synchronized void uninstall()
	{
		//System.out.println("UNINSTALL CALLED");

		this.myKeyboardFocusManager.removeVetoableChangeListener(
				"focusedWindow", this.windowFocusPreventer); //$NON-NLS-1$

		this.myTopWindow.removeComponentListener(this.mouseBlockerSynchronizer);

		if (this.myTopWindow instanceof RootPaneContainer)
		{
			//System.out.println("removing from RootPaneContainer");
			JLayeredPane l = ((RootPaneContainer) this.myTopWindow).getLayeredPane();
			l.remove(this.mouseBlocker);
		}
		else
		{
			//System.out.println("EMERGENCY REMOVE");
			// better than nothing...
			this.myTopWindow.setEnabled(true);
		}

		this.myTopWindow.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));

		//System.out.println("UNINSTALL COMPLETED");
	}

	// =========================================================================
	// Some final variables which are custom classes.
	// =========================================================================
	final Component mouseBlocker = new JPanel()
	{
		{
			setOpaque(false);
			enableEvents(AWTEvent.MOUSE_EVENT_MASK);
		}

		@Override
		protected void processMouseEvent(MouseEvent event)
		{
			/*
			if (logger.isLoggable(Level.FINE))
			{
				String type;
				switch (event.getID())
				{
				case MouseEvent.MOUSE_CLICKED:
					type = "MOUSE_CLICKED"; //$NON-NLS-1$
					break;
				case MouseEvent.MOUSE_PRESSED:
					type = "MOUSE_PRESSED"; //$NON-NLS-1$
					break;
				case MouseEvent.MOUSE_RELEASED:
					type = "MOUSE_RELEASED"; //$NON-NLS-1$
					break;
				case MouseEvent.MOUSE_ENTERED:
					type = "MOUSE_ENTERED"; //$NON-NLS-1$
					break;
				case MouseEvent.MOUSE_EXITED:
					type = "MOUSE_EXITED"; //$NON-NLS-1$
					break;
				default:
					type = "(unknown)"; //$NON-NLS-1$
					break;
				}
			}
			*/
			event.consume();
		}
	};

	private final ComponentListener mouseBlockerSynchronizer = new ComponentAdapter()
	{
		@Override
		public void componentResized(ComponentEvent event)
		{
			//System.out.println("Parent window resized");
			ModalProgressDialogFlex.this.mouseBlocker.setSize(event.getComponent().getSize());
		}
	};

	/**
	 * Prevents window from getting focus.
	 */
	private final VetoableChangeListener windowFocusPreventer = new VetoableChangeListener()
	{
		public void vetoableChange(PropertyChangeEvent event)
				throws PropertyVetoException
		{
			String property = event.getPropertyName();
			//System.out.println("vetoableChange: " + property);
			if (property.equals("focusedWindow")) //$NON-NLS-1$
			{
				//System.out.println("focusedWindow property changed");
				if (event.getNewValue() == ModalProgressDialogFlex.this.myTopWindow)
				{
					EventQueue.invokeLater(this.refocuser);
					synchronized (ModalProgressDialogFlex.this.FINISHED_SYNC)
					{
						if (ModalProgressDialogFlex.this.iAmFinished == false) { throw new PropertyVetoException(
								"Window may not take focus" //$NON-NLS-1$
										+ " while it has a modal dialog showing", //$NON-NLS-1$
								event); }
					}
				}
			}
		}

		private final Runnable refocuser = new Runnable()
		{
			public void run()
			{
				ModalProgressDialogFlex.this.myDialog.requestFocus();
			}
		};
	};

	// =========================================================================
	// Test Stub
	// =========================================================================
	public static void main(String[] args)
	{
		final BoundedRangeModel range = new DefaultBoundedRangeModel();
		range.setMaximum(10);

		JButton hitMe = new JButton("Hit Me!"); //$NON-NLS-1$
		final JLabel wacked = new JLabel("Not Wacked Yet"); //$NON-NLS-1$
		JButton start = new JButton("Start"); //$NON-NLS-1$
		final JLabel countLbl = new JLabel("Inactive at moment"); //$NON-NLS-1$

		JPanelRows contents = new JPanelRows();
		JPanel row = contents.topRow();
		row.add(hitMe);
		row.add(wacked);

		row = contents.nextRow();
		row.add(start);
		row.add(countLbl);

		final JFrame testFrame = new JFrame("File Tree Test"); //$NON-NLS-1$
		testFrame.getContentPane().add(new JScrollPane(contents));
		testFrame.setSize(300, 300);
		testFrame.addWindowListener(new WindowAdapter()
		{
			@Override
			public void windowClosing(WindowEvent evt)
			{
				System.exit(0);
			}
		});
		org.gerhardb.lib.swing.SwingUtils.centerOnScreen(testFrame);

		hitMe.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				if (wacked.getBackground().equals(Color.GREEN))
				{
					wacked.setBackground(Color.RED);
					wacked.setOpaque(true);
					wacked.setForeground(Color.YELLOW);
					wacked.setText("WACKED"); //$NON-NLS-1$
				}
				else
				{
					wacked.setBackground(Color.GREEN);
					wacked.setOpaque(true);
					wacked.setForeground(Color.BLUE);
					wacked.setText("unwacked at moment"); //$NON-NLS-1$
				}
			}
		});

		class ShowProgress
		{
			ShowProgress(final String msg)
			{
				SwingUtilities.invokeLater(new Runnable()
				{
					public void run()
					{
						countLbl.setText(msg);
					}
				});
			}
		}

		class TestMoveIt implements Runnable
		{
			private TestMoveIt()
			{
		   	// Don't allow public creation.
			}

			public void run()
			{
				int length = range.getMaximum() + 1; // CAREFULL
				for (int i = 0; i < length; i++)
				{
					range.setValue(i);
					new ShowProgress("Doing: " + i); //$NON-NLS-1$
					try
					{
						Thread.sleep(1000L);
					}
					catch (InterruptedException e)
					{
						break;
					}
					// This does not seem to work even when I interrupt the
					// thread.
					if (Thread.currentThread().isInterrupted())
					{
						//System.out.println("THREAD WAS INTERRUPTED");
						break;
					}
				}
				new ShowProgress("All Done!"); //$NON-NLS-1$
				//System.out.println("All Done!");

			}
		}

		start.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				TestMoveIt moveIt = new TestMoveIt();
				Thread runMe = new Thread(moveIt);

				ModalProgressDialogFlex flex = new ModalProgressDialogFlex(
						testFrame, "Moving Files", //$NON-NLS-1$
						"The files are being moved...", //$NON-NLS-1$
						"Stop Move", //$NON-NLS-1$
						range, 2000);
				flex.run(runMe);
			}
		});

		testFrame.setVisible(true);
	}
}

/*

EventQueue.invokeLater(new Runnable()
{
   public void run()
   {
   }
});
*/