/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.util.app;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.DisplayMode;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.print.Printable;
import java.net.URLDecoder;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.prefs.Preferences;

import javax.print.Doc;
import javax.print.DocFlavor;
import javax.print.DocPrintJob;
import javax.print.PrintException;
import javax.print.PrintService;
import javax.print.PrintServiceLookup;
import javax.print.ServiceUI;
import javax.print.SimpleDoc;
import javax.print.attribute.Attribute;
import javax.print.attribute.DocAttributeSet;
import javax.print.attribute.HashDocAttributeSet;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.ScrollPaneConstants;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.print.PrintableText;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.startup.AppPluginStartup;
import org.gerhardb.lib.util.startup.AppPreferencesFactory;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 * Info Display.
 */
public class Info extends JFrame
{
	private static int MEGABYTE = 1024 * 1024;
	private static String EOL = "<br>\n"; //$NON-NLS-1$
	private static String TAB = "     "; //$NON-NLS-1$

	JEditorPane myEditorPane = new JEditorPane();
	Manifest myManifest = AboutBox.readManifest();

	/*
	 * =========================================================================
	 *                         Constructor
	 * =========================================================================
	 */

	/**
	 * Prints basic Information Report.
	 */
	public Info(JFrame frame, InfoPlugins plugins)
	{
      this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());
      
		Dimension size = new Dimension(700, 400);
		setSize(size);

		setTitle("JIBS Report"); //$NON-NLS-1$

		this.myEditorPane.setEditable(false);
		this.myEditorPane.setDisabledTextColor(Color.black);
		this.myEditorPane.setEditorKit(JEditorPane
				.createEditorKitForContentType("text/html")); //$NON-NLS-1$

		String jibsVersion = AboutBox.getJibsVersion(this.myManifest);
		
		StringBuffer sb = new StringBuffer();
		sb.append("<html><body><center><strong><big><big>JIBS Report" //$NON-NLS-1$
				+ "</big><br>" + new Date()  //$NON-NLS-1$
				+ "<br>Version: " + jibsVersion  //$NON-NLS-1$
				+ "</big></strong></center>"); //$NON-NLS-1$
		category(sb, "Image Support"); //$NON-NLS-1$
		imageReport(sb);
		category(sb, "Startup Properties"); //$NON-NLS-1$
		startupProperties(sb);
		category(sb, "Manifest"); //$NON-NLS-1$
		manifest(sb);
		category(sb, "Runtime"); //$NON-NLS-1$
		runtimeReport(sb);
		category(sb, "Internationalization"); //$NON-NLS-1$
		localizationReport(sb);
		category(sb, "Printers"); //$NON-NLS-1$
		printerListReport(sb);
		category(sb, "Graphics"); //$NON-NLS-1$
		graphicsReport(sb, frame);
		category(sb, "Viewer"); //$NON-NLS-1$
		plugins.info(sb);
		category(sb, "System"); //$NON-NLS-1$
		systemReport(sb);
		category(sb, "Thread Groups"); //$NON-NLS-1$
		threadGroupReport(Thread.currentThread().getThreadGroup(), sb);
		//category(sb, "Printer Details"); //$NON-NLS-1$
		//printerReport(sb);
		sb.append("</body></html>"); //$NON-NLS-1$

		this.myEditorPane.setText(sb.toString());
		this.myEditorPane.setCaretPosition(0);

		JScrollPane center = new JScrollPane(this.myEditorPane,
				ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
				ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);

		getContentPane().setLayout(new BorderLayout());
		getContentPane().add(center, BorderLayout.CENTER);

		this.setJMenuBar(this.makeMenus());

		SwingUtils.centerOnScreen(this);

		super.setVisible(true);
	}
	
	JMenuBar makeMenus()
	{
		JMenu fileMenu = new JMenu("File"); //$NON-NLS-1$
		JMenu editMenu = new JMenu("Edit"); //$NON-NLS-1$

		JMenuItem closeMenu = new JMenuItem("Close"); // myEditorPane //$NON-NLS-1$
		closeMenu.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				Info.this.dispose();
			}
		});

		JMenuItem saveMenu = new JMenuItem("Save As..."); //$NON-NLS-1$
		saveMenu.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				new PrintableText(Info.this.myEditorPane).file();
			}
		});

		JMenuItem printPreviewMenu = new JMenuItem(Jibs.getString("print.preview") + "..."); //$NON-NLS-1$ //$NON-NLS-2$
		printPreviewMenu.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				PrintableText pt = new PrintableText(Info.this.myEditorPane);
				pt.previewHTML(Info.this);
			}
		});

		JMenuItem printImageMenu = new JMenuItem("Print...", KeyEvent.VK_P); //$NON-NLS-1$
		printImageMenu.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				new PrintableText(Info.this.myEditorPane).printContent();
			}
		});

		JMenuItem printTextMenu = new JMenuItem("Print Text...", KeyEvent.VK_T); //$NON-NLS-1$
		printTextMenu.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				printWithService();
			}
		});

		JMenuItem copyMenu = new JMenuItem("Copy All"); //$NON-NLS-1$
		copyMenu.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				new PrintableText(Info.this.myEditorPane).copy();
			}
		});

		fileMenu.add(saveMenu);
		fileMenu.add(printPreviewMenu);
		fileMenu.add(printImageMenu);
		//fileMenu.add(printTextMenu);
		fileMenu.add(new JSeparator());
		fileMenu.add(closeMenu);

		editMenu.add(copyMenu);

		JMenuBar menuBar = new JMenuBar();
		menuBar.add(fileMenu);
		menuBar.add(editMenu);
		//menuBar.add( Box.createHorizontalGlue() );
		//menuBar.add( helpMenu );

		return menuBar;
	} // makeMenus()

	/**
	 * Attempt to use the Java Print Service.
	 * Unfortunately, the server printable produces same output.
	 * http://java.sun.com/j2se/1.4.1/docs/guide/jps/spec/JPSTOC.fm.html
	 */
	public void printWithService()
	{
		Printable printable = new PrintableText(this.myEditorPane);
		DocFlavor flavor = DocFlavor.SERVICE_FORMATTED.PRINTABLE;

		PrintRequestAttributeSet pras = new HashPrintRequestAttributeSet();
		//pras.add(new Copies(1));
		//pras.add(MediaSize.getMediaSizeForName(MediaSizeName.ISO_C6));
		//pras.add(Sides.DUPLEX);
		PrintService[] allServices = PrintServiceLookup.lookupPrintServices(
				flavor, pras);
		if (allServices.length == 0)
		{
			// TO DO - INFORM USER!
			System.out.println("No print services found"); //$NON-NLS-1$
			return;
		}
		PrintService defaultService = PrintServiceLookup
				.lookupDefaultPrintService();
		PrintService service = ServiceUI.printDialog(null, 200, 200, allServices,
				defaultService, flavor, pras);
		if (service == null)
		{
			System.out.println("User cancelled print"); //$NON-NLS-1$
			return;
		}
		DocPrintJob job = service.createPrintJob();
		DocAttributeSet das = new HashDocAttributeSet();
		//Doc doc = new SimpleDoc(htmlIn, flavor, das);
		//Doc doc = new SimpleDoc(htmlIn, flavor, das);
		Doc doc = new SimpleDoc(printable, flavor, das);
		try
		{
			job.print(doc, pras);
		}
		catch (PrintException pe)
		{
			pe.printStackTrace();
		}
	}

	/**
	 * Attempt to use the Java Print Service.
	 * Unfortunately, HTML does not seem to be implemented in the
	 * current version of java.
	 * http://java.sun.com/j2se/1.4.1/docs/guide/jps/spec/JPSTOC.fm.html
	 * http://access1.sun.com/tools/JPSDemo/jpsdemo.html
	 * http://javaalmanac.com/egs/javax.print/pkg.html
	 * http://forum.java.sun.com/thread.jsp?thread=146439&forum=31&message=1185367
	 * http://java.sun.com/products/java-media/2D/forDevelopers/sdk12print.html
	 *
	 * RECOMMENDED SITE TO TRY FOR NEXT TIME:
	 * http://java.sun.com/products/java-media/2D/forDevelopers/sdk12print.html
	 *
	 * Two other approaches:
	 * a) convert html to postscript using open source widget
	 *    and print postscript.
	 * b) java help seems to be able to do this.  Bundle with distribution.
	 *    License looks like it would work.
	 */
	/*
	 DocFlavor flavor = DocFlavor.STRING.TEXT_HTML;
	 DocFlavor flavor = DocFlavor.STRING.TEXT_PLAIN;
	 DocFlavor flavor = DocFlavor.INPUT_STREAM.TEXT_HTML_UTF_8;
	 */
	/*
	void printHtml(String html)
	{
		DocFlavor flavor = DocFlavor.INPUT_STREAM.TEXT_HTML_US_ASCII;
		InputStream htmlIn = stringToInputStream(html);

		PrintRequestAttributeSet pras = new HashPrintRequestAttributeSet();
		//pras.add(new Copies(1));
		//pras.add(MediaSize.getMediaSizeForName(MediaSizeName.ISO_C6));
		//pras.add(Sides.DUPLEX);
		PrintService[] allServices = PrintServiceLookup.lookupPrintServices(
				flavor, pras);
		if (allServices.length == 0)
		{
			// TO DO - INFORM USER!
			System.out.println("No print services found"); //$NON-NLS-1$
			return;
		}
		PrintService defaultService = PrintServiceLookup
				.lookupDefaultPrintService();
		PrintService service = ServiceUI.printDialog(null, 200, 200, allServices,
				defaultService, flavor, pras);
		if (service == null)
		{
			System.out.println("User cancelled print"); //$NON-NLS-1$
			return;
		}
		DocPrintJob job = service.createPrintJob();
		DocAttributeSet das = new HashDocAttributeSet();
		//Doc doc = new SimpleDoc(html, flavor, das);
		Doc doc = new SimpleDoc(htmlIn, flavor, das);
		try
		{
			job.print(doc, pras);
		}
		catch (PrintException pe)
		{
			pe.printStackTrace();
		}
	}
	
	// from impx.src.com.saic.isd.printscreen.Util
	InputStream stringToInputStream(String convertMe)
	{
		// Other way would be to wrap the a StringRead as a InputStream.
		byte[] bytes = null;
		try
		{
			bytes = convertMe.getBytes("UTF-8"); //$NON-NLS-1$
		}
		catch (java.io.UnsupportedEncodingException ex)
		{
			// We know it ain't gonna happen!
			ex.printStackTrace();
		}
		return new ByteArrayInputStream(bytes);
	}
*/
	
	// ==========================================================================
	// Info
	// ==========================================================================

	void category(StringBuffer sb, String title)
	{
		sb.append("<br><hr><center><strong><big>"); //$NON-NLS-1$
		sb.append(title);
		sb.append("</big></strong></center><hr>");		 //$NON-NLS-1$
	}

	void localizationReport(StringBuffer sb)
	{
		Locale locale = Locale.getDefault();
		sb.append("Display Country: " + locale.getDisplayCountry() + EOL); //$NON-NLS-1$
		sb.append("Display Language: " + locale.getDisplayLanguage() + EOL); //$NON-NLS-1$
		sb.append("Display Name: " + locale.getDisplayName() + EOL); //$NON-NLS-1$
		sb.append("Display Variant: " + locale.getDisplayVariant() + EOL); //$NON-NLS-1$
		sb.append("ISO3 Country: " + locale.getISO3Country() + EOL); //$NON-NLS-1$
		sb.append("ISO3 Language: " + locale.getISO3Language() + EOL); //$NON-NLS-1$
		sb.append("Country: " + locale.getCountry() + EOL); //$NON-NLS-1$
		sb.append("Language: " + locale.getLanguage() + EOL); //$NON-NLS-1$
		sb.append("Variant: " + locale.getVariant() + EOL); //$NON-NLS-1$
		sb.append("Locale: " + locale.toString() + EOL); //$NON-NLS-1$
	}

	void startupProperties(StringBuffer sb)
	{
		sb.append("Started: " + AppStarter.TIME_STARTED + EOL);
		sb.append("Jibs Starup Directory: " + AppStarter.getStartupDirectory() + EOL);
		if (AppStarter.isThumbdrive())
		{
			sb.append("Thumbdrive Option USED" + EOL);			
		}
		else
		{
			sb.append("Thumbdrive Option NOT used" + EOL);		
		}		
		sb.append("Java Prefs: " + Preferences.userRoot().getClass() + EOL);
		sb.append("Pref File: " + System.getProperty(AppPreferencesFactory.PROP_FILE_LOOKUP) + EOL);
		sb.append("Run Commmand: " + AppPluginStartup.getRunCommand() + EOL);		
	}

	void manifest(StringBuffer sb)
	{
		Manifest manifest = AboutBox.readManifest();
		if (manifest != null)
		{
			Attributes attr = manifest.getMainAttributes();
			sb.append("Specification Title: " + attr.getValue(Attributes.Name.SPECIFICATION_TITLE) + EOL);
			sb.append("Specification Version: " + attr.getValue(Attributes.Name.SPECIFICATION_VERSION) + EOL);
			sb.append("Specification Vendor: " + attr.getValue(Attributes.Name.SPECIFICATION_VENDOR) + EOL);
			sb.append("Implementation Vendor: " + attr.getValue(Attributes.Name.IMPLEMENTATION_VENDOR) + EOL);
			sb.append("Implementation Vendor ID: " + attr.getValue(Attributes.Name.IMPLEMENTATION_VENDOR_ID) + EOL);
			sb.append("Implementation Title: " + attr.getValue(Attributes.Name.IMPLEMENTATION_TITLE) + EOL);
			sb.append("Implementation Version: " + attr.getValue(Attributes.Name.IMPLEMENTATION_VERSION) + EOL);
			sb.append("Implementation URL: " + attr.getValue(Attributes.Name.IMPLEMENTATION_URL) + EOL);
			sb.append("Main Class: " + attr.getValue(Attributes.Name.MAIN_CLASS) + EOL);
			
			sb.append("Built by: " + attr.getValue("Built-By") + EOL);
		}
	}
	
	void runtimeReport(StringBuffer sb)
	{
		Runtime rt = Runtime.getRuntime();
		sb.append("Java: "); //$NON-NLS-1$
		sb.append(System.getProperty("java.version")); //$NON-NLS-1$
		sb.append(EOL);
		sb.append("Available Processors: "); //$NON-NLS-1$
		sb.append(rt.availableProcessors());
		sb.append(EOL);
		sb.append("Free Memory: ").append((int) rt.freeMemory() / MEGABYTE); //$NON-NLS-1$
		sb.append(" Megabytes"); //$NON-NLS-1$
		sb.append(EOL);
		sb.append("Total Memory: ").append((int) rt.totalMemory() / MEGABYTE); //$NON-NLS-1$
		sb.append(" Megabytes"); //$NON-NLS-1$
		sb.append(EOL);
		sb.append("Max Memory (set by -Xmx): "); //$NON-NLS-1$
		sb.append((int) rt.maxMemory() / MEGABYTE);
		sb.append(" Megabytes" ); //$NON-NLS-1$
		sb.append(EOL);
		sb.append("Preferences Class: "); //$NON-NLS-1$
		sb.append(Preferences.userRoot().getClass().toString()); //$NON-NLS-1$
		sb.append(EOL);
		sb.append("Property File Used: "); //$NON-NLS-1$
		sb.append(System.getProperty(AppPreferencesFactory.PROP_FILE_LOOKUP)); //$NON-NLS-1$
		sb.append(EOL);
		sb.append("Startup Directory: "); //$NON-NLS-1$
		sb.append(AppStarter.getStartupDirectory().getAbsolutePath()); //$NON-NLS-1$
		sb.append(EOL);
		sb.append("Protection Domain: "); //$NON-NLS-1$
		try
		{
			sb.append(URLDecoder.decode(Info.class.getProtectionDomain().getCodeSource().getLocation().getPath(), "UTF-8")); //$NON-NLS-1$
		}
		catch(Exception ex)
      {
      	ex.printStackTrace();
      }
		sb.append(EOL);
		if (AppStarter.isThumbdrive())
		{
			sb.append("Thumbdrive Option Selected"); //$NON-NLS-1$			
		}
		else
		{
			sb.append("No Thumbdrive Option"); //$NON-NLS-1$			
		}
		sb.append(EOL);
	}

	void systemReport(StringBuffer sb)
	{
		sb.append("System Properties:\n"); //$NON-NLS-1$
		propertyReport(System.getProperties(), sb);
	}

	void threadGroupReport(ThreadGroup group, StringBuffer buff)
	{
		if (buff == null) { return; }
		if (group == null)
		{
			buff.append("No thread group"); //$NON-NLS-1$
			return;
		}

		int threadSize = group.activeCount();
		Thread[] threads = new Thread[threadSize];
		group.enumerate(threads);

		int groupSize = group.activeGroupCount();
		ThreadGroup[] groups = new ThreadGroup[groupSize];
		group.enumerate(groups);

		buff.append("Thead Group: ").append(group.getName()).append(EOL); //$NON-NLS-1$
		buff.append("Date: ").append(new Date()).append(EOL); //$NON-NLS-1$
		buff.append("Parent: ").append(group.getParent()).append(EOL); //$NON-NLS-1$
		buff.append("Thead Count: ").append(threadSize).append(EOL); //$NON-NLS-1$
		buff.append("Group Count: ").append(groupSize).append(EOL); //$NON-NLS-1$
		buff.append(EOL);

		buff.append("Theads:\n"); //$NON-NLS-1$
		buff.append("<ol>\n"); //$NON-NLS-1$
		for (int i = 0; i < threadSize; i++)
		{
			Thread aThread = threads[i];
			if (aThread != null)
			{
				buff.append("<li>").append(aThread.getName()).append("</li>\n"); //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
		buff.append("</ol>\n"); //$NON-NLS-1$

		buff.append("Thead Groups:\n"); //$NON-NLS-1$
		buff.append("<ol>\n"); //$NON-NLS-1$
		for (int i = 0; i < groupSize; i++)
		{
			ThreadGroup aGroup = groups[i];
			if (aGroup != null)
			{
				buff.append("<li>"); //$NON-NLS-1$
				threadGroupReport(aGroup, buff);
				buff.append("</li>\n"); //$NON-NLS-1$
			}
		}
		buff.append("</ol>\n"); //$NON-NLS-1$
	}

	void propertyReport(Properties showMe, StringBuffer sb)
	{
		Object[] items = showMe.entrySet().toArray();
		sb.append("<ol>\n"); //$NON-NLS-1$
		for (int i = 0; i < items.length; i++)
		{
			Map.Entry<?,?> entry = (Map.Entry<?,?>) items[i];
			sb.append("<li>\n"); //$NON-NLS-1$
			sb.append(entry.getKey()).append("=").append(entry.getValue()); //$NON-NLS-1$
			sb.append("</li>\n"); //$NON-NLS-1$
		}
		sb.append("</ol>\n"); //$NON-NLS-1$
	}

	void printerListReport(StringBuffer sb)
	{
		PrintService[] theServices = PrintServiceLookup.lookupPrintServices(null,
				null);
		for (int i = 0; i < theServices.length; i++)
		{
			sb.append("Print Service #"); //$NON-NLS-1$
			sb.append(i);
			sb.append(": " ); //$NON-NLS-1$
			sb.append(theServices[i].getName());
			sb.append(EOL);
		}
	}
	
	/*
	void printerReport(StringBuffer sb) 
	{
		PrintService[] theServices = PrintServiceLookup.lookupPrintServices(null,
				null);
		for (int i = 0; i < theServices.length; i++)
		{
			sb.append("<strong>Print Service #" + i + ": "  //$NON-NLS-1$ //$NON-NLS-2$
					+ theServices[i].getName() + "</strong>" + EOL); //$NON-NLS-1$
			DocFlavor[] flavors = theServices[i].getSupportedDocFlavors();
			for (int j = 0; j < flavors.length; j++)
			{
				sb.append("=========================================="); //$NON-NLS-1$
				sb.append(EOL);
				DocFlavor aFlavor = flavors[j];
				sb.append("toString: " + aFlavor.toString()); //$NON-NLS-1$
				sb.append(EOL);
				sb.append("Mime: " + aFlavor.getMimeType()); //$NON-NLS-1$
				sb.append(EOL);
				sb.append("Media: " + aFlavor.getMediaType()); //$NON-NLS-1$
				sb.append(EOL);
				sb.append("Sub: " + aFlavor.getMediaSubtype()); //$NON-NLS-1$
				sb.append(EOL);
				sb.append("Rep: " + aFlavor.getRepresentationClassName()); //$NON-NLS-1$
				sb.append(EOL);
				sb.append("Class name: " + aFlavor.getClass().getName()); //$NON-NLS-1$
				sb.append(EOL);
			}
		}
	}
	*/
	
	void imageReport(StringBuffer sb)
	{
		String[] plugins = ImageFactory.getImageFactory().getImagePluginsListing();
		for (int i = 0; i < plugins.length; i++)
		{
			sb.append(plugins[i] + EOL);
		}
		
		sb.append(EOL);
		
		String[] endings = ImageFactory.getImageFactory().getFilter().getEndings();
		sb.append(Jibs.getString("Info.1"));				 //$NON-NLS-1$
		for (int i = 0; i < endings.length; i++)
		{
			if (i>0)
			{
				sb.append(", ");				 //$NON-NLS-1$
			}
			sb.append(endings[i]);
		}
		sb.append(EOL);
		
		String[] pluginsNot = ImageFactory.getImageFactory().getImagePluginsNotLoadedListing();
		sb.append(EOL);
		if (pluginsNot.length == 0)
		{
			sb.append("All available plug-ins were loaded. ");				 
		}
		else
		{
			sb.append("Plug-ins which were not loaded: ");				 
			for (int i = 0; i < pluginsNot.length; i++)
			{
				if (i>0)
				{
					sb.append(", ");				 //$NON-NLS-1$
				}
				sb.append(pluginsNot[i]);
			}
		}
		sb.append(EOL);
	}
	
	void graphicsReport(StringBuffer sb, JFrame frame)
	{
		sb.append("Displayed JIBS frame size: " 
				+ frame.getWidth() + "W " + frame.getHeight() + "H     " + EOL); 
		
		sb.append("Displayed JIBS frame upper left hand corner: " 
			   + frame.getX() + "X " + frame.getY() + "Y     "+ EOL);
		
		GraphicsEnvironment ge = 
			GraphicsEnvironment.getLocalGraphicsEnvironment();

		sb.append("GraphicsEnvironment: " + ge + EOL); //$NON-NLS-1$
		sb.append("Default Screen Device: "  //$NON-NLS-1$
				+  ge.getDefaultScreenDevice() + EOL);

		GraphicsDevice[] sd = ge.getScreenDevices();
		for (int i = 0; i < sd.length; i++)
		{
			sb.append("Graphics Device #" + i + ": " + sd[i] + "<br>"); //$NON-NLS-1$ //$NON-NLS-2$
			if (sd[i].isFullScreenSupported())
			{
				sb.append(TAB + "FullScreenSupported: yes" + EOL); //$NON-NLS-1$
			}
			else
			{
				sb.append(TAB + "FullScreenSupported: no" + EOL); //$NON-NLS-1$
			}
			if (sd[i].isDisplayChangeSupported())
			{
				sb.append(TAB + "DisplayChangeSupported: yes" + EOL); //$NON-NLS-1$
			}
			else
			{
				sb.append(TAB + "DisplayChangeSupported: no" + EOL); //$NON-NLS-1$
			}
			DisplayMode ddm = sd[i].getDisplayMode();
			sb.append(TAB + "Default Display Mode: width " + ddm.getWidth() //$NON-NLS-1$
					+ "   height " + ddm.getHeight() + "   bits " //$NON-NLS-1$ //$NON-NLS-2$
					+ ddm.getBitDepth() + "   refresh " + ddm.getRefreshRate()  //$NON-NLS-1$
					+ EOL);
			/*
			DisplayMode[] dm = sd[i].getDisplayModes();
			for (int j = 0; j < dm.length; j++)
			{
				sb.append("Mode: width " + dm[i].getWidth()
						+ "   height " + dm[j].getHeight() + "   bits "
						+ dm[j].getBitDepth() + "   refresh "
						+ dm[j].getRefreshRate() + "<br>/n");
			}
			*/
		}		
	}

	/*
	static void printerReportNew(StringBuffer sb)
	{
		PrintService[] theServices = PrintServiceLookup.lookupPrintServices(null,
				null);
		for (int i = 0; i < theServices.length; i++)
		{
			sb.append("<strong>Print Service #" + i + ": "  //$NON-NLS-1$ //$NON-NLS-2$
					+ theServices[i].getName() + "</strong>" + EOL); //$NON-NLS-1$
			if (theServices[i].getName().equals("Lexmark 5400 Series")) //$NON-NLS-1$
			{
				
			}
		}
	}		
	 */
	
	// ==========================================================================
	// Main
	// ==========================================================================
	@SuppressWarnings("unchecked")
	public static void main(String[] args)
	{
		//String html = "<head><body>Hello, World!</body></head>";
		//Info.printHtml(html);
		
		//new Info(new LoadingShow());
		PrintService[] theServices = PrintServiceLookup.lookupPrintServices(null,
				null);
		for (int i = 0; i < theServices.length; i++)
		{
			System.out.println("Print Service #" + i + ": " + theServices[i].getName());  //$NON-NLS-1$ //$NON-NLS-2$
			if (theServices[i].getName().equals("Lexmark 5400 Series")) //$NON-NLS-1$
			{
				PrintService aService = theServices[i];
				Class[] classes = aService.getSupportedAttributeCategories();
				for (int j = 0; j < classes.length; j++)
				{
					System.out.println("Class #" + j + ": " + classes[j].getName()); 	 //$NON-NLS-1$ //$NON-NLS-2$
					if ( classes[j].getName().equals("javax.print.attribute.standard.PrinterResolution")) //$NON-NLS-1$
					{
						Class aClass = classes[j];
						System.out.println(aService.getDefaultAttributeValue(aClass)); 							
					}
				}
				Attribute[] attributes = aService.getAttributes().toArray();
				for (int j = 0; j < attributes.length; j++)
				{
					System.out.println("Attribute #" + j + ": " + attributes[j].getName() + "     "  + attributes[j].getClass()); 					 //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				}
			}
		}
	}
}
