/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.util.startup;

import java.io.*;

import javax.swing.JOptionPane;

import com.saic.isd.util.ArgList;


/**
 * Sets up classpath and then calls JIBS.
 * Used for thumbdrives and checking for plugins.
 */
public class AppPluginStartup
{
	public static final String JIBS_OUT = "JIBS OUT";
	
	private static final String DOUBLE_QUOTE = "\"";
	private static final String SPACE = " ";
	
	private static final String JIBS_PREFERENCES_PARAMETER = "-Djava.util.prefs.PreferencesFactory=org.gerhardb.lib.util.startup.AppPreferencesFactory";
	private static final String PATH_SEPARATOR = System.getProperty("path.separator"); //$NON-NLS-1$
	private static final String JAVA_COMMAND = 
		System.getProperty("java.home") + File.separator + "bin" + File.separator + "java" + SPACE;
	
	private static String clsRunCommand = "Started JIBS Directly";
	private static String clsPropsFileName;
	
	private static boolean iFoundPlugins = false;
	
	// ==========================================================================
	// Public
	// ==========================================================================
	public static String getRunCommand()
	{
		return clsRunCommand;
	}
	
	// ==========================================================================
	// Private
	// ==========================================================================
	
	public static void startupAppWithPlugins(String mainClass, String propsFileName, String[] args)
	{
		clsPropsFileName = propsFileName;
		
		/*
		//File logFileName = new File("/media/win_d/dev/workspaces/deliver/log.txt");		
		File logFileName = new File("D:/dev/workspaces/deliver this/log.txt");		
		try
		{
	     PrintWriter writer =
	          new PrintWriter(
	          new BufferedWriter(
	          new FileWriter( logFileName ) ) );
	     AppStarter.clsLogStream = writer;
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
		*/
		
		AppStarter.log("##################################################"); //$NON-NLS-1$
		AppStarter.log("Jibs Plugin Startup Booting Jibs"); //$NON-NLS-1$
		AppStarter.log("##################################################"); //$NON-NLS-1$

		StringBuffer runMe = new StringBuffer(300);		
		runMe.append(JAVA_COMMAND);
		
		String propFilePath = loadPreferences(args);
		if (propFilePath != null)
		{
			AppStarter.log("USING PREFERENCES FROM JIBS PROPERTY FILE!"); //$NON-NLS-1$				
			runMe.append(JIBS_PREFERENCES_PARAMETER);
		}
		
		String memory = getMemoryParameter(args);
		if (memory != null)
		{
			runMe.append(memory);			
		}
		
		runMe.append(" -classpath ");
		runMe.append(DOUBLE_QUOTE);
		runMe.append(getClasspaths());
		runMe.append(DOUBLE_QUOTE);
		runMe.append(SPACE);
		runMe.append(mainClass);
		runMe.append(SPACE);
		runMe.append(strippedArgsToString(args));

		if (iFoundPlugins)
		{
			AppStarter.log("PLUGINS FOUND!"); //$NON-NLS-1$			
		}
		
		if (propFilePath != null)
		{
			runMe.append(SPACE);
			runMe.append(propFilePath);	
		}
		
		AppStarter.log("STARTING WITH COMMAND LINE!"); //$NON-NLS-1$
		clsRunCommandLine(runMe.toString());		
	}

	// ==========================================================================
	// protected
	// ==========================================================================
	private static String loadPreferences(String[] args)
	{
		ArgList argList = new ArgList(args);
		//AppStarter.log("JibsPluginStartup.loadPreferences()");
		File propFile = null;
		if (argList.isOption(AppStarter.PROPS))
		{
			String propFileName = argList.getOption(AppStarter.PROPS);
			AppStarter.log("JibsPluginStartup Loading Jibs PROPS from: " + propFileName);
			if (propFileName == null || propFileName.length() == 0)
			{
		      JOptionPane.showMessageDialog(
		            null,
		            "No prop file specified \n"
		            + "\nLoading JIBS with standard settings",
		            "JIBS Start Problem",
		            JOptionPane.ERROR_MESSAGE );			
				return null;
			}
			
			propFile = new File(propFileName);
			
			// We need either a good property file
			// or a good directory.
			boolean propFileOK = propFile.exists();
			if (!propFileOK)
			{
				File parent = propFile.getParentFile();
				if (parent != null)
				{
					propFileOK = parent.exists();		
				}
			}
			if (propFileOK)
			{
				propFileName = propFile.getAbsolutePath();
				//AppStarter.log("JibsPluginStartup.PropFile[PROPS]: " + propFileName);
				//System.setProperty(JibsPreferencesFactory.PROP_FILE_LOOKUP, propFileName);
				return AppStarter.PROPS + SPACE + DOUBLE_QUOTE + propFileName + DOUBLE_QUOTE;
			}
			JOptionPane.showMessageDialog(
			      null,
			      "Invalid path: \n" + propFileName
			      + "\nLoading JIBS with standard settings",
			      "JIBS Start Problem",
			      JOptionPane.ERROR_MESSAGE );			
			return null;
		}
		else if (argList.isOption(AppStarter.THUMBDRIVE))
		{
			File startDir = AppStarter.getStartupDirectory();
			AppStarter.log("Loading Jibs THUMBDRIVE from: " + startDir);
			if (startDir != null)
			{
				// For thumbdrives, we will load up even if the file is not there.
				// We can write to it later, if its not read only.
				// If it is read only (CD-ROM, DVD), oh well, at least JIBS came up.
				String propFileName = startDir.getAbsolutePath() + "/" + clsPropsFileName;
				//AppStarter.log("JibsPluginStartup.PropFile[THUMBDRIVE]: " + propFileName);
				//System.setProperty(JibsPreferencesFactory.PROP_FILE_LOOKUP, propFileName);
				return AppStarter.THUMBDRIVE + SPACE + DOUBLE_QUOTE + propFileName + DOUBLE_QUOTE;
			}
		}
		else
		{
			AppStarter.log("JibsPluginStartup Checking Jibs LOCAL DRIVE");
			File startDir = AppStarter.getStartupDirectory();
			if (startDir != null)
			{
				File[] startupDirList = startDir.listFiles();
				if (startupDirList != null)
				{
					for (int i = 0; i < startupDirList.length; i++)
					{
						if (startupDirList[i].getName().equalsIgnoreCase(clsPropsFileName)) //$NON-NLS-1$
						{
							String propFileName = startupDirList[i].getAbsolutePath();
							//AppStarter.log("JibsPluginStartup.PropFile[LOCAL DRIVE]: " + propFileName);
							//System.setProperty(JibsPreferencesFactory.PROP_FILE_LOOKUP, propFileName);
							return AppStarter.PROPS + SPACE + DOUBLE_QUOTE + propFileName + DOUBLE_QUOTE;
						}
					}
				}
			}
		}		
		return null;
	}
	
	/**
	 * Consume -props and -thumbdrive
	 * @param args
	 * @return
	 */
	private static String strippedArgsToString(String[] args)
	{
		// 
		StringBuffer buff = new StringBuffer(100);
		for(int i=0; i<args.length; i++)
		{
			if(AppStarter.PROPS.equalsIgnoreCase((args[i])))
			{
				i++;
			}
			else if(AppStarter.THUMBDRIVE.equalsIgnoreCase((args[i])))
			{
				//do nothing
			}
			else
			{
				buff.append(' ');
				buff.append(DOUBLE_QUOTE);
				buff.append(args[i]);
				buff.append(DOUBLE_QUOTE);
			}
		}
		return buff.toString();
	}
	
	private static String getClasspaths()
	{	
		StringBuffer pluginBuff = new StringBuffer(200);
		
		// Find out where jar file is.
		File starupDir = AppStarter.getStartupDirectory();
		File[] startupDirList = starupDir.listFiles();
		if (startupDirList != null)
		{
			for (int i = 0; i < startupDirList.length; i++)
			{
				if (startupDirList[i].getName().endsWith("ij.jar")) //$NON-NLS-1$
				{
					pluginBuff.append(PATH_SEPARATOR + startupDirList[i].getAbsolutePath());
				}
			}
		}		
		
		// ImageJ now incorporated.
		// Left in in case technique needed for other plugins.
		/*
		if (pluginBuff.length() > 0)
		{
			iFoundPlugins = true;
		}
		*/
		
		String classpaths = System.getProperty("java.class.path"); //$NON-NLS-1$
		if (classpaths != null)
		{
			if (iFoundPlugins)
			{
				return classpaths + pluginBuff.toString();
			}
			return classpaths;
		}
		
		// Not sure its possible to get to here.
		if (iFoundPlugins)
		{
			// Get rid of the first path separator.
			return pluginBuff.toString().substring(1);
		}
		return null;
	}
	
	private static String getMemoryParameter(String[] args)
	{
		String version = System.getProperty("java.version"); //$NON-NLS-1$
		if (!version.startsWith("1.6")) //$NON-NLS-1$
		{
			AppStarter.log("Starting Legacy JIBS (anything not Java 1.6)"); //$NON-NLS-1$
			
			StringBuffer allArgs = new StringBuffer(300);
			for(int i=0; i<args.length; i++)
			{
				allArgs.append(args[i]);
			}
			
			// this did now work well
			StringBuffer javaBuff = new StringBuffer(30);
			if (allArgs.indexOf("-Xms") == -1)
			{
				javaBuff.append(" -Xms40M");
			}
			if (allArgs.indexOf("-Xmx") == -1)
			{
				javaBuff.append(" -Xmx100M");
			}
			
			if (javaBuff.length() > 0)
			{
				return javaBuff.toString();
			}
			return null;
		}
		AppStarter.log("Starting JIBS for Java 1.6"); //$NON-NLS-1$
		return null;
	}
	
	private static void clsRunCommandLine(String runMe)
	{
		clsRunCommand = runMe;
		Runtime runtime = Runtime.getRuntime();
		try
		{
			Process process = runtime.exec(runMe);
			InputStream is = process.getInputStream();
			InputStreamReader isr = new InputStreamReader(is);
			BufferedReader br = new BufferedReader(isr);
			String line;

			AppStarter.log("##################################################"); //$NON-NLS-1$
			AppStarter.log("Switching output to runtime"); //$NON-NLS-1$
			AppStarter.log("Run Command: \n" + runMe);
			AppStarter.log("##################################################"); //$NON-NLS-1$

			// Print commands received from process to screen.
			try
			{
				while ((line = br.readLine()) != null)
				{
					if (JIBS_OUT.equals(line))
					{
						System.exit(0);
					}
					// DO NOT PUT EXPLANATION AT FRONT OF THIS.
					// It is just a pass through.
					AppStarter.log(line);
				}
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
			catch(Throwable t)
			{
				t.printStackTrace();
			}
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}		
	}
	
	// ==========================================================================
	// MAIN
	// ==========================================================================
	public static void testArgs(String[] args)
	{
		System.out.println(args.length);
		AppStarter.log(strippedArgsToString( new String[]{}));
		AppStarter.log(strippedArgsToString( new String[]{"-props", "sample"}));
		AppStarter.log(strippedArgsToString( new String[]{"-thumbdrive"}));
		AppStarter.log(strippedArgsToString( new String[]{"-props", "sample", "-view", "home"}));
		AppStarter.log(strippedArgsToString( new String[]{"-thumbdrive", "-view", "home"}));
	}
}