/*
 * Copyright (c) 2011 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.print;

import java.awt.*;
import java.awt.font.TextLayout;
import java.awt.image.BufferedImage;

/**
 * PrintableBaseTest
 */
class PrintTestPattern
{
	private static final float dash1[] = { 10.0f }; // size, not directly pixel related.
	private static final float DEFAULT_STROKE = 1.0f; // Java default line	

	private BasicStroke myStroke;
	private BasicStroke myDashed;

	// Maximum printable margins
	InteriorLayout interior;

	/**
	 * PrintTestPattern
	 */
	PrintTestPattern(InteriorLayout interiorIn)
	{
		this.interior = interiorIn;

		this.myStroke = new BasicStroke(DEFAULT_STROKE);
		this.myDashed = new BasicStroke(DEFAULT_STROKE, BasicStroke.CAP_BUTT,
				BasicStroke.JOIN_MITER, 10.0f, dash1, 0.0f);
	}

	public void drawBasicTestPattern(Graphics g)
	{
		//BasicStroke thickDashed = new BasicStroke(5.0f, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 10.0f, dash1, 0.0f);

		// Draw a red rectangle exactly at the margin.
		g.setColor(Color.RED);
		//g.setStroke(this.myStroke);
		g.drawRect(this.interior.getLeftMargin(), this.interior.getTopMargin(),
				this.interior.getWidth(), this.interior.getHeight());

		// Draw a rectangle just inside the margin.
		g.setColor(Color.BLACK);
		//g.setStroke(this.myDashed);
		g.drawRect(this.interior.getLeftMargin() + 1, this.interior
				.getTopMargin() + 1, this.interior.getWidth() - 2, this.interior
				.getHeight() - 2);

		g.setColor(Color.RED);
		//g.setStroke(this.myStroke);

		// Draw a line down the middle of the page.
		g.drawLine(this.interior.getCenterX(), this.interior.getTopMargin(),
				this.interior.getCenterX(), this.interior.getBottomMargin());

		// Draw a like across the middle of the page.
		g.drawLine(this.interior.getLeftMargin(), this.interior.getCenterY(),
				this.interior.getRightMargin(), this.interior.getCenterY());

		// Now a pair of diagonal lines.
		g.setColor(Color.BLUE);
		g.drawLine(this.interior.getLeftMargin(), this.interior.getTopMargin(),
				this.interior.getRightMargin(), this.interior.getBottomMargin());
		g.drawLine(this.interior.getRightMargin(), this.interior.getTopMargin(),
				this.interior.getLeftMargin(), this.interior.getBottomMargin());

		// Sample Header & Footer
		g.setColor(Color.BLACK);
		g.setFont(new Font("SansSerif", Font.PLAIN, 12));
		g.drawString("Sample Header in Upper Left Corner", this.interior
				.getLeftMargin() + 5, this.interior.getTopMargin() + 40);

		// PUT SAMPLE HEADER WHERE IT SHOULD REALLY GO
		// Image is just to get a Graphics2D.
		BufferedImage buff = new BufferedImage(this.interior.getWidth(),
				this.interior.getHeight(), BufferedImage.TYPE_INT_RGB);
		Graphics2D g2 = buff.createGraphics();
		Font headerFont = new Font("SansSerif", Font.ITALIC, 14);
		TextLayout text = new TextLayout(
				"Pgy", headerFont, g2.getFontRenderContext()); //$NON-NLS-1$
		int headerHeight = (int) (text.getAscent() + text.getDescent());
		int bottomOfText = this.interior.getTopMargin() + headerHeight;

		// Draw a line just underneath the text.
		g.drawLine(this.interior.getLeftMargin(), bottomOfText, this.interior
				.getRightMargin(), bottomOfText);

		// Text is draw up from the Ascent and down from the Descent.
		// This puts text at the top of the printable margin.
		int textX = (int) (this.interior.getTopMargin() + text.getAscent());

		// ------------- Actual Printing Follows -----------------------
		g.setColor(Color.BLUE);
		String printString = "The Centered Header Sample";
		// Right Justified
		text = new TextLayout(printString, headerFont, g2.getFontRenderContext());
		float headerWidth = (float) text.getBounds().getWidth();
		int textY = (int) (this.interior.getLeftMargin() + ((this.interior
				.getWidth() - headerWidth) / 2));
		//text.draw(g, textY, textX);
		g.drawString(printString, textY, textX);
	}

	public void drawBasicTestPattern(Graphics2D g2)
	{
		//BasicStroke thickDashed = new BasicStroke(5.0f, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 10.0f, dash1, 0.0f);

		// Draw a red rectangle exactly at the margin.
		g2.setPaint(Color.RED);
		g2.setStroke(this.myStroke);
		g2.drawRect(this.interior.getLeftMargin(), this.interior.getTopMargin(),
				this.interior.getWidth(), this.interior.getHeight());

		// Draw a rectangle just inside the margin.
		g2.setPaint(Color.BLACK);
		g2.setStroke(this.myDashed);
		g2.drawRect(this.interior.getLeftMargin() + 1, this.interior
				.getTopMargin() + 1, this.interior.getWidth() - 2, this.interior
				.getHeight() - 2);

		g2.setPaint(Color.RED);
		g2.setStroke(this.myStroke);

		// Draw a line down the middle of the page.
		g2.drawLine(this.interior.getCenterX(), this.interior.getTopMargin(),
				this.interior.getCenterX(), this.interior.getBottomMargin());

		// Draw a like across the middle of the page.
		g2.drawLine(this.interior.getLeftMargin(), this.interior.getCenterY(),
				this.interior.getRightMargin(), this.interior.getCenterY());

		// Now a pair of diagonal lines.
		g2.setPaint(Color.BLUE);
		g2.drawLine(this.interior.getLeftMargin(), this.interior.getTopMargin(),
				this.interior.getRightMargin(), this.interior.getBottomMargin());
		g2.drawLine(this.interior.getRightMargin(), this.interior.getTopMargin(),
				this.interior.getLeftMargin(), this.interior.getBottomMargin());

		// Sample Header & Footer
		g2.setPaint(Color.BLACK);
		g2.setFont(new Font("SansSerif", Font.PLAIN, 12));
		g2.drawString("Sample Header in Upper Left Corner", this.interior
				.getLeftMargin() + 5, this.interior.getTopMargin() + 40);

		// PUT SAMPLE HEADER WHERE IT SHOULD REALLY GO
		Font headerFont = new Font("SansSerif", Font.ITALIC, 14);
		TextLayout text = new TextLayout(
				"Pgy", headerFont, g2.getFontRenderContext()); //$NON-NLS-1$
		int headerHeight = (int) (text.getAscent() + text.getDescent());
		int bottomOfText = this.interior.getTopMargin() + headerHeight;

		// Draw a line just underneath the text.
		g2.drawLine(this.interior.getLeftMargin(), bottomOfText, this.interior
				.getRightMargin(), bottomOfText);

		// Text is draw up from the Ascent and down from the Descent.
		// This puts text at the top of the printable margin.
		int textX = (int) (this.interior.getTopMargin() + text.getAscent());

		// ------------- Actual Printing Follows -----------------------
		g2.setPaint(Color.BLUE);
		String printString = "The Centered Header Sample";
		// Right Justified
		text = new TextLayout(printString, headerFont, g2.getFontRenderContext());
		float headerWidth = (float) text.getBounds().getWidth();
		int textY = (int) (this.interior.getLeftMargin() + ((this.interior
				.getWidth() - headerWidth) / 2));
		text.draw(g2, textY, textX);
	}
}
