/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.io.file;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.DosFileAttributeView;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFileAttributes;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Iterator;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.file.AbstractTempDirTest;
import org.apache.commons.io.file.PathUtils;
import org.apache.commons.io.file.TempDirectory;
import org.apache.commons.io.file.TempFile;
import org.apache.commons.io.filefilter.NameFileFilter;
import org.apache.commons.io.test.TestUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.SystemProperties;
import org.apache.commons.lang3.SystemUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Test;

public class PathUtilsTest
extends AbstractTempDirTest {
    private static final String STRING_FIXTURE = "Hello World";
    private static final byte[] BYTE_ARRAY_FIXTURE = "Hello World".getBytes(StandardCharsets.UTF_8);
    private static final String TEST_JAR_NAME = "test.jar";
    private static final String TEST_JAR_PATH = "src/test/resources/org/apache/commons/io/test.jar";
    private static final String PATH_FIXTURE = "NOTICE.txt";

    private Path current() {
        return PathUtils.current();
    }

    private Long getLastModifiedMillis(Path file) throws IOException {
        return Files.getLastModifiedTime(file, new LinkOption[0]).toMillis();
    }

    private Path getNonExistentPath() {
        return Paths.get("/does not exist/for/certain", new String[0]);
    }

    private FileSystem openArchive(Path p, boolean createNew) throws IOException {
        if (createNew) {
            HashMap<String, String> env = new HashMap<String, String>();
            env.put("create", "true");
            URI fileUri = p.toAbsolutePath().toUri();
            URI uri = URI.create("jar:" + fileUri.toASCIIString());
            return FileSystems.newFileSystem(uri, env, null);
        }
        return FileSystems.newFileSystem(p, (ClassLoader)null);
    }

    private void setLastModifiedMillis(Path file, long millis) throws IOException {
        Files.setLastModifiedTime(file, FileTime.fromMillis(millis));
    }

    @Test
    public void testCopyDirectoryForDifferentFilesystemsWithAbsolutePath() throws IOException {
        Path archivePath = Paths.get(TEST_JAR_PATH, new String[0]);
        try (FileSystem archive = this.openArchive(archivePath, false);){
            Path sourceDir = archive.getPath("dir1", new String[0]);
            PathUtils.copyDirectory(sourceDir, this.tempDirPath, new CopyOption[0]);
            Assertions.assertTrue((boolean)Files.exists(this.tempDirPath.resolve("f1"), new LinkOption[0]));
            sourceDir = archive.getPath("/next", new String[0]);
            PathUtils.copyDirectory(sourceDir, this.tempDirPath, new CopyOption[0]);
            Assertions.assertTrue((boolean)Files.exists(this.tempDirPath.resolve("dir"), new LinkOption[0]));
        }
    }

    @Test
    public void testCopyDirectoryForDifferentFilesystemsWithAbsolutePathReverse() throws IOException {
        try (FileSystem archive = this.openArchive(this.tempDirPath.resolve(TEST_JAR_NAME), true);){
            Path targetDir = archive.getPath("target", new String[0]);
            Files.createDirectory(targetDir, new FileAttribute[0]);
            Path sourceDir = Paths.get("src/test/resources/org/apache/commons/io/dirs-2-file-size-2", new String[0]).toAbsolutePath();
            PathUtils.copyDirectory(sourceDir, targetDir, new CopyOption[0]);
            Assertions.assertTrue((boolean)Files.exists(targetDir.resolve("dirs-a-file-size-1"), new LinkOption[0]));
            targetDir = archive.getPath("/", new String[0]);
            PathUtils.copyDirectory(sourceDir, targetDir, new CopyOption[0]);
            Assertions.assertTrue((boolean)Files.exists(targetDir.resolve("dirs-a-file-size-1"), new LinkOption[0]));
        }
    }

    @Test
    public void testCopyDirectoryForDifferentFilesystemsWithRelativePath() throws IOException {
        Path archivePath = Paths.get(TEST_JAR_PATH, new String[0]);
        try (FileSystem archive = this.openArchive(archivePath, false);
             FileSystem targetArchive = this.openArchive(this.tempDirPath.resolve(TEST_JAR_NAME), true);){
            Path targetDir = targetArchive.getPath("targetDir", new String[0]);
            Files.createDirectory(targetDir, new FileAttribute[0]);
            Path sourceDir = archive.getPath("next", new String[0]);
            PathUtils.copyDirectory(sourceDir, targetDir, new CopyOption[0]);
            Assertions.assertTrue((boolean)Files.exists(targetDir.resolve("dir"), new LinkOption[0]));
            sourceDir = archive.getPath("/dir1", new String[0]);
            PathUtils.copyDirectory(sourceDir, targetDir, new CopyOption[0]);
            Assertions.assertTrue((boolean)Files.exists(targetDir.resolve("f1"), new LinkOption[0]));
        }
    }

    @Test
    public void testCopyDirectoryForDifferentFilesystemsWithRelativePathReverse() throws IOException {
        try (FileSystem archive = this.openArchive(this.tempDirPath.resolve(TEST_JAR_NAME), true);){
            Path targetDir = archive.getPath("target", new String[0]);
            Files.createDirectory(targetDir, new FileAttribute[0]);
            Path sourceDir = Paths.get("src/test/resources/org/apache/commons/io/dirs-2-file-size-2", new String[0]);
            PathUtils.copyDirectory(sourceDir, targetDir, new CopyOption[0]);
            Assertions.assertTrue((boolean)Files.exists(targetDir.resolve("dirs-a-file-size-1"), new LinkOption[0]));
            targetDir = archive.getPath("/", new String[0]);
            PathUtils.copyDirectory(sourceDir, targetDir, new CopyOption[0]);
            Assertions.assertTrue((boolean)Files.exists(targetDir.resolve("dirs-a-file-size-1"), new LinkOption[0]));
        }
    }

    @Test
    public void testCopyFile() throws IOException {
        Path sourceFile = Paths.get("src/test/resources/org/apache/commons/io/dirs-1-file-size-1/file-size-1.bin", new String[0]);
        Path targetFile = PathUtils.copyFileToDirectory(sourceFile, this.tempDirPath, new CopyOption[0]);
        Assertions.assertTrue((boolean)Files.exists(targetFile, new LinkOption[0]));
        Assertions.assertEquals((long)Files.size(sourceFile), (long)Files.size(targetFile));
    }

    @Test
    public void testCopyURL() throws IOException {
        Path sourceFile = Paths.get("src/test/resources/org/apache/commons/io/dirs-1-file-size-1/file-size-1.bin", new String[0]);
        URL url = new URL("file:///" + FilenameUtils.getPath(sourceFile.toAbsolutePath().toString()) + sourceFile.getFileName());
        Path targetFile = PathUtils.copyFileToDirectory(url, this.tempDirPath, new CopyOption[0]);
        Assertions.assertTrue((boolean)Files.exists(targetFile, new LinkOption[0]));
        Assertions.assertEquals((long)Files.size(sourceFile), (long)Files.size(targetFile));
    }

    @Test
    public void testCreateDirectoriesAlreadyExists() throws IOException {
        Assertions.assertEquals((Object)this.tempDirPath.getParent(), (Object)PathUtils.createParentDirectories(this.tempDirPath, new FileAttribute[0]));
    }

    @Test
    public void testCreateDirectoriesForRoots() throws IOException {
        for (Path path : FileSystems.getDefault().getRootDirectories()) {
            Path parent = path.getParent();
            Assertions.assertNull((Object)parent);
            Assertions.assertEquals((Object)parent, (Object)PathUtils.createParentDirectories(path, new FileAttribute[0]));
        }
    }

    @Test
    public void testCreateDirectoriesForRootsLinkOptionNull() throws IOException {
        for (File f : File.listRoots()) {
            Path path = f.toPath();
            Assertions.assertEquals((Object)path.getParent(), (Object)PathUtils.createParentDirectories(path, (LinkOption)null, new FileAttribute[0]));
        }
    }

    @Test
    public void testCreateDirectoriesNew() throws IOException {
        Assertions.assertEquals((Object)this.tempDirPath, (Object)PathUtils.createParentDirectories(this.tempDirPath.resolve("child"), new FileAttribute[0]));
    }

    @Test
    public void testCreateDirectoriesSymlink() throws IOException {
        Path symlinkedDir = PathUtilsTest.createTempSymbolicLinkedRelativeDir(this.tempDirPath);
        String leafDirName = "child";
        Path newDirFollowed = PathUtils.createParentDirectories(symlinkedDir.resolve("child"), PathUtils.NULL_LINK_OPTION, new FileAttribute[0]);
        Assertions.assertEquals((Object)Files.readSymbolicLink(symlinkedDir), (Object)newDirFollowed);
    }

    @Test
    public void testCreateDirectoriesSymlinkClashing() throws IOException {
        Path symlinkedDir = PathUtilsTest.createTempSymbolicLinkedRelativeDir(this.tempDirPath);
        Assertions.assertEquals((Object)symlinkedDir, (Object)PathUtils.createParentDirectories(symlinkedDir.resolve("child"), new FileAttribute[0]));
    }

    @Test
    public void testGetBaseNamePathBaseCases() {
        Assertions.assertEquals((Object)"bar", (Object)PathUtils.getBaseName(Paths.get("a/b/c/bar.foo", new String[0])));
        Assertions.assertEquals((Object)"foo", (Object)PathUtils.getBaseName(Paths.get("foo", new String[0])));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getBaseName(Paths.get("", new String[0])));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getBaseName(Paths.get(".", new String[0])));
        for (File f : File.listRoots()) {
            Assertions.assertNull((Object)PathUtils.getBaseName(f.toPath()));
        }
        if (SystemUtils.IS_OS_WINDOWS) {
            Assertions.assertNull((Object)PathUtils.getBaseName(Paths.get("C:\\", new String[0])));
        }
    }

    @Test
    public void testGetBaseNamePathCornerCases() {
        Assertions.assertNull((Object)PathUtils.getBaseName(null));
        Assertions.assertEquals((Object)"foo", (Object)PathUtils.getBaseName(Paths.get("foo.", new String[0])));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getBaseName(Paths.get("bar/.foo", new String[0])));
    }

    @Test
    public void testGetDosFileAttributeView() {
        DosFileAttributeView dosFileAttributeView = PathUtils.getDosFileAttributeView(this.current(), new LinkOption[0]);
        Path path = Paths.get("this-file-does-not-exist-at.all", new String[0]);
        Assertions.assertFalse((boolean)Files.exists(path, new LinkOption[0]));
        if (SystemUtils.IS_OS_MAC) {
            Assertions.assertNull((Object)dosFileAttributeView);
            Assertions.assertNull((Object)PathUtils.getDosFileAttributeView(path, new LinkOption[0]));
        } else {
            Assertions.assertNotNull((Object)dosFileAttributeView);
            Assertions.assertNotNull((Object)PathUtils.getDosFileAttributeView(path, new LinkOption[0]));
        }
        Assertions.assertThrows(NullPointerException.class, () -> PathUtils.getDosFileAttributeView(null, new LinkOption[0]));
    }

    @Test
    public void testGetExtension() {
        Assertions.assertNull((Object)PathUtils.getExtension(null));
        Assertions.assertEquals((Object)"ext", (Object)PathUtils.getExtension(Paths.get("file.ext", new String[0])));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getExtension(Paths.get("README", new String[0])));
        Assertions.assertEquals((Object)"com", (Object)PathUtils.getExtension(Paths.get("domain.dot.com", new String[0])));
        Assertions.assertEquals((Object)"jpeg", (Object)PathUtils.getExtension(Paths.get("image.jpeg", new String[0])));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getExtension(Paths.get("a.b/c", new String[0])));
        Assertions.assertEquals((Object)"txt", (Object)PathUtils.getExtension(Paths.get("a.b/c.txt", new String[0])));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getExtension(Paths.get("a/b/c", new String[0])));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getExtension(Paths.get("a.b\\c", new String[0])));
        Assertions.assertEquals((Object)"txt", (Object)PathUtils.getExtension(Paths.get("a.b\\c.txt", new String[0])));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getExtension(Paths.get("a\\b\\c", new String[0])));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getExtension(Paths.get("C:\\temp\\foo.bar\\README", new String[0])));
        Assertions.assertEquals((Object)"ext", (Object)PathUtils.getExtension(Paths.get("../filename.ext", new String[0])));
        if (File.separatorChar != '\\') {
            Assertions.assertEquals((Object)"txt", (Object)PathUtils.getExtension(Paths.get("foo.exe:bar.txt", new String[0])));
        }
    }

    @Test
    public void testGetFileName() {
        Assertions.assertNull(PathUtils.getFileName(null, null));
        Assertions.assertNull((Object)PathUtils.getFileName(null, Path::toString));
        Assertions.assertNull((Object)PathUtils.getFileName(Paths.get("/", new String[0]), Path::toString));
        Assertions.assertNull((Object)PathUtils.getFileName(Paths.get("/", new String[0]), Path::toString));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getFileName(Paths.get("", new String[0]), Path::toString));
        Assertions.assertEquals((Object)"a", (Object)PathUtils.getFileName(Paths.get("a", new String[0]), Path::toString));
        Assertions.assertEquals((Object)"a", (Object)PathUtils.getFileName(Paths.get("p", "a"), Path::toString));
    }

    @Test
    public void testGetFileNameString() {
        Assertions.assertNull((Object)PathUtils.getFileNameString(Paths.get("/", new String[0])));
        Assertions.assertEquals((Object)"", (Object)PathUtils.getFileNameString(Paths.get("", new String[0])));
        Assertions.assertEquals((Object)"a", (Object)PathUtils.getFileNameString(Paths.get("a", new String[0])));
        Assertions.assertEquals((Object)"a", (Object)PathUtils.getFileNameString(Paths.get("p", "a")));
    }

    @Test
    public void testGetLastModifiedFileTime_File_Present() throws IOException {
        Assertions.assertNotNull((Object)PathUtils.getLastModifiedFileTime(this.current().toFile()));
    }

    @Test
    public void testGetLastModifiedFileTime_Path_Absent() throws IOException {
        Assertions.assertNull((Object)PathUtils.getLastModifiedFileTime(this.getNonExistentPath(), new LinkOption[0]));
    }

    @Test
    public void testGetLastModifiedFileTime_Path_FileTime_Absent() throws IOException {
        FileTime fromMillis = FileTime.fromMillis(0L);
        Assertions.assertEquals((Object)fromMillis, (Object)PathUtils.getLastModifiedFileTime(this.getNonExistentPath(), fromMillis, new LinkOption[0]));
    }

    @Test
    public void testGetLastModifiedFileTime_Path_Present() throws IOException {
        Assertions.assertNotNull((Object)PathUtils.getLastModifiedFileTime(this.current(), new LinkOption[0]));
    }

    @Test
    public void testGetLastModifiedFileTime_URI_Present() throws IOException {
        Assertions.assertNotNull((Object)PathUtils.getLastModifiedFileTime(this.current().toUri()));
    }

    @Test
    public void testGetLastModifiedFileTime_URL_Present() throws IOException, URISyntaxException {
        Assertions.assertNotNull((Object)PathUtils.getLastModifiedFileTime(this.current().toUri().toURL()));
    }

    @Test
    public void testGetTempDirectory() {
        Path tempDirectory = Paths.get(SystemProperties.getJavaIoTmpdir(), new String[0]);
        Assertions.assertEquals((Object)tempDirectory, (Object)PathUtils.getTempDirectory());
    }

    @Test
    public void testIsDirectory() throws IOException {
        Assertions.assertFalse((boolean)PathUtils.isDirectory(null, new LinkOption[0]));
        Assertions.assertTrue((boolean)PathUtils.isDirectory(this.tempDirPath, new LinkOption[0]));
        try (TempFile testFile1 = TempFile.create(this.tempDirPath, "prefix", null, new FileAttribute[0]);){
            Assertions.assertFalse((boolean)PathUtils.isDirectory(testFile1.get(), new LinkOption[0]));
            Path ref = null;
            try (TempDirectory tempDir = TempDirectory.create(this.getClass().getCanonicalName(), new FileAttribute[0]);){
                ref = tempDir.get();
                Assertions.assertTrue((boolean)PathUtils.isDirectory(tempDir.get(), new LinkOption[0]));
            }
            Assertions.assertFalse((boolean)PathUtils.isDirectory(ref, new LinkOption[0]));
        }
    }

    @Test
    public void testIsPosix() throws IOException {
        boolean isPosix;
        try {
            Files.getPosixFilePermissions(this.current(), new LinkOption[0]);
            isPosix = true;
        }
        catch (UnsupportedOperationException e) {
            isPosix = false;
        }
        Assertions.assertEquals((Object)isPosix, (Object)PathUtils.isPosix(this.current(), new LinkOption[0]));
    }

    @Test
    public void testIsPosixAbsentFile() {
        Assertions.assertFalse((boolean)PathUtils.isPosix(Paths.get("ImNotHereAtAllEver.never", new String[0]), new LinkOption[0]));
        Assertions.assertFalse((boolean)PathUtils.isPosix(null, new LinkOption[0]));
    }

    @Test
    public void testIsRegularFile() throws IOException {
        Assertions.assertFalse((boolean)PathUtils.isRegularFile(null, new LinkOption[0]));
        Assertions.assertFalse((boolean)PathUtils.isRegularFile(this.tempDirPath, new LinkOption[0]));
        try (TempFile testFile1 = TempFile.create(this.tempDirPath, "prefix", null, new FileAttribute[0]);){
            Assertions.assertTrue((boolean)PathUtils.isRegularFile(testFile1.get(), new LinkOption[0]));
            Files.delete(testFile1.get());
            Assertions.assertFalse((boolean)PathUtils.isRegularFile(testFile1.get(), new LinkOption[0]));
        }
    }

    @Test
    public void testNewDirectoryStream() throws Exception {
        NameFileFilter pathFilter = new NameFileFilter(PATH_FIXTURE);
        try (DirectoryStream<Path> stream = PathUtils.newDirectoryStream(this.current(), pathFilter);){
            Iterator<Path> iterator = stream.iterator();
            Path path = iterator.next();
            Assertions.assertEquals((Object)PATH_FIXTURE, (Object)PathUtils.getFileNameString(path));
            Assertions.assertFalse((boolean)iterator.hasNext());
        }
    }

    @Test
    public void testNewOutputStreamExistingFileAppendFalse() throws IOException {
        this.testNewOutputStreamNewFile(false);
        this.testNewOutputStreamNewFile(false);
    }

    @Test
    public void testNewOutputStreamExistingFileAppendTrue() throws IOException {
        this.testNewOutputStreamNewFile(true);
        Path file = this.writeToNewOutputStream(true);
        Assertions.assertArrayEquals((byte[])ArrayUtils.addAll((byte[])BYTE_ARRAY_FIXTURE, (byte[])BYTE_ARRAY_FIXTURE), (byte[])Files.readAllBytes(file));
    }

    public void testNewOutputStreamNewFile(boolean append) throws IOException {
        Path file = this.writeToNewOutputStream(append);
        Assertions.assertArrayEquals((byte[])BYTE_ARRAY_FIXTURE, (byte[])Files.readAllBytes(file));
    }

    @Test
    public void testNewOutputStreamNewFileAppendFalse() throws IOException {
        this.testNewOutputStreamNewFile(false);
    }

    @Test
    public void testNewOutputStreamNewFileAppendTrue() throws IOException {
        this.testNewOutputStreamNewFile(true);
    }

    @Test
    public void testNewOutputStreamNewFileInsideExistingSymlinkedDir() throws IOException {
        Path symlinkDir = PathUtilsTest.createTempSymbolicLinkedRelativeDir(this.tempDirPath);
        Path file = symlinkDir.resolve("test.txt");
        OutputStream outputStream = PathUtils.newOutputStream(file, new LinkOption[0], new OpenOption[0]);
        if (outputStream != null) {
            outputStream.close();
        }
        if ((outputStream = PathUtils.newOutputStream(file, null, new OpenOption[0])) != null) {
            outputStream.close();
        }
        if ((outputStream = PathUtils.newOutputStream(file, true)) != null) {
            outputStream.close();
        }
        if ((outputStream = PathUtils.newOutputStream(file, false)) != null) {
            outputStream.close();
        }
    }

    @Test
    public void testReadAttributesPosix() throws IOException {
        boolean isPosix;
        try {
            Files.getPosixFilePermissions(this.current(), new LinkOption[0]);
            isPosix = true;
        }
        catch (UnsupportedOperationException e) {
            isPosix = false;
        }
        Assertions.assertEquals((Object)isPosix, (Object)(PathUtils.readAttributes(this.current(), PosixFileAttributes.class, new LinkOption[0]) != null ? 1 : 0));
    }

    @Test
    public void testReadStringEmptyFile() throws IOException {
        Path path = Paths.get("src/test/resources/org/apache/commons/io/test-file-empty.bin", new String[0]);
        Assertions.assertEquals((Object)"", (Object)PathUtils.readString(path, StandardCharsets.UTF_8));
        Assertions.assertEquals((Object)"", (Object)PathUtils.readString(path, null));
    }

    @Test
    public void testReadStringSimpleUtf8() throws IOException {
        Path path = Paths.get("src/test/resources/org/apache/commons/io/test-file-simple-utf8.bin", new String[0]);
        String expected = "ABC\r\n";
        Assertions.assertEquals((Object)"ABC\r\n", (Object)PathUtils.readString(path, StandardCharsets.UTF_8));
        Assertions.assertEquals((Object)"ABC\r\n", (Object)PathUtils.readString(path, null));
    }

    @Test
    public void testSetReadOnlyFile() throws IOException {
        Path resolved = this.tempDirPath.resolve("testSetReadOnlyFile.txt");
        boolean isPosix = PathUtils.isPosix(this.tempDirPath, new LinkOption[0]);
        Assumptions.assumeFalse((boolean)SystemUtils.IS_OS_LINUX);
        PathUtils.writeString(resolved, "test", StandardCharsets.UTF_8, new OpenOption[0]);
        boolean readable = Files.isReadable(resolved);
        boolean writable = Files.isWritable(resolved);
        boolean regularFile = Files.isRegularFile(resolved, new LinkOption[0]);
        boolean executable = Files.isExecutable(resolved);
        boolean hidden = Files.isHidden(resolved);
        boolean directory = Files.isDirectory(resolved, new LinkOption[0]);
        boolean symbolicLink = Files.isSymbolicLink(resolved);
        Assertions.assertTrue((boolean)readable);
        Assertions.assertTrue((boolean)writable);
        PathUtils.setReadOnly(resolved, false, new LinkOption[0]);
        Assertions.assertTrue((boolean)Files.isReadable(resolved), (String)"isReadable");
        Assertions.assertTrue((boolean)Files.isWritable(resolved), (String)"isWritable");
        PathUtils.setReadOnly(resolved, false, new LinkOption[0]);
        Assertions.assertTrue((boolean)Files.isReadable(resolved), (String)"isReadable");
        Assertions.assertTrue((boolean)Files.isWritable(resolved), (String)"isWritable");
        Assertions.assertEquals((Object)regularFile, (Object)Files.isReadable(resolved));
        Assertions.assertEquals((Object)executable, (Object)Files.isExecutable(resolved));
        Assertions.assertEquals((Object)hidden, (Object)Files.isHidden(resolved));
        Assertions.assertEquals((Object)directory, (Object)Files.isDirectory(resolved, new LinkOption[0]));
        Assertions.assertEquals((Object)symbolicLink, (Object)Files.isSymbolicLink(resolved));
        PathUtils.setReadOnly(resolved, true, new LinkOption[0]);
        if (isPosix) {
            Assertions.assertFalse((boolean)Files.isReadable(resolved), (String)"isReadable");
        } else {
            Assertions.assertTrue((boolean)Files.isReadable(resolved), (String)"isReadable");
        }
        Assertions.assertFalse((boolean)Files.isWritable(resolved), (String)"isWritable");
        DosFileAttributeView dosFileAttributeView = PathUtils.getDosFileAttributeView(resolved, new LinkOption[0]);
        if (dosFileAttributeView != null) {
            Assertions.assertTrue((boolean)dosFileAttributeView.readAttributes().isReadOnly());
        }
        if (isPosix) {
            Assertions.assertFalse((boolean)Files.isReadable(resolved));
        } else {
            Assertions.assertEquals((Object)regularFile, (Object)Files.isReadable(resolved));
        }
        Assertions.assertEquals((Object)executable, (Object)Files.isExecutable(resolved));
        Assertions.assertEquals((Object)hidden, (Object)Files.isHidden(resolved));
        Assertions.assertEquals((Object)directory, (Object)Files.isDirectory(resolved, new LinkOption[0]));
        Assertions.assertEquals((Object)symbolicLink, (Object)Files.isSymbolicLink(resolved));
        PathUtils.setReadOnly(resolved, false, new LinkOption[0]);
        PathUtils.deleteFile(resolved);
    }

    @Test
    public void testSetReadOnlyFileAbsent() {
        Assertions.assertThrows(IOException.class, () -> PathUtils.setReadOnly(Paths.get("does-not-exist-at-all-ever-never", new String[0]), true, new LinkOption[0]));
    }

    @Test
    public void testTouch() throws IOException {
        Assertions.assertThrows(NullPointerException.class, () -> FileUtils.touch(null));
        Path file = this.managedTempDirPath.resolve("touch.txt");
        Files.deleteIfExists(file);
        Assertions.assertFalse((boolean)Files.exists(file, new LinkOption[0]), (String)"Bad test: test file still exists");
        PathUtils.touch(file);
        Assertions.assertTrue((boolean)Files.exists(file, new LinkOption[0]), (String)"touch() created file");
        try (OutputStream out = Files.newOutputStream(file, new OpenOption[0]);){
            Assertions.assertEquals((long)0L, (long)Files.size(file), (String)"Created empty file.");
            out.write(0);
        }
        Assertions.assertEquals((long)1L, (long)Files.size(file), (String)"Wrote one byte to file");
        long y2k = new GregorianCalendar(2000, 0, 1).getTime().getTime();
        this.setLastModifiedMillis(file, y2k);
        Assertions.assertEquals((long)y2k, (Long)this.getLastModifiedMillis(file), (String)"Bad test: set lastModified set incorrect value");
        long nowMillis = System.currentTimeMillis();
        PathUtils.touch(file);
        Assertions.assertEquals((long)1L, (long)Files.size(file), (String)"FileUtils.touch() didn't empty the file.");
        Assertions.assertNotEquals((long)y2k, (Long)this.getLastModifiedMillis(file), (String)"FileUtils.touch() changed lastModified");
        int delta = 3000;
        Assertions.assertTrue((this.getLastModifiedMillis(file) >= nowMillis - 3000L ? 1 : 0) != 0, (String)"FileUtils.touch() changed lastModified to more than now-3s");
        Assertions.assertTrue((this.getLastModifiedMillis(file) <= nowMillis + 3000L ? 1 : 0) != 0, (String)"FileUtils.touch() changed lastModified to less than now+3s");
    }

    @Test
    public void testWriteStringToFile1() throws Exception {
        Path file = this.tempDirPath.resolve("write.txt");
        PathUtils.writeString(file, "Hello /u1234", StandardCharsets.UTF_8, new OpenOption[0]);
        byte[] text = "Hello /u1234".getBytes(StandardCharsets.UTF_8);
        TestUtils.assertEqualContent(text, file);
    }

    private Path writeToNewOutputStream(boolean append) throws IOException {
        Path file = this.tempDirPath.resolve("test1.txt");
        try (OutputStream os = PathUtils.newOutputStream(file, append);){
            os.write(BYTE_ARRAY_FIXTURE);
        }
        return file;
    }
}

