/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.io.input;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.MemoryMappedFileInputStream;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.RandomUtils;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.provider.ValueSource;

public class MemoryMappedFileInputStreamTest {
    @TempDir
    Path tempDir;

    @AfterEach
    public void afterEach() {
        System.gc();
        Thread.yield();
        System.runFinalization();
        Thread.yield();
        System.gc();
        Thread.yield();
        System.runFinalization();
        Thread.yield();
    }

    private Path createTestFile(int size) throws IOException {
        return Files.write(Files.createTempFile(this.tempDir, null, null, new FileAttribute[0]), RandomUtils.nextBytes((int)size), new OpenOption[0]);
    }

    private MemoryMappedFileInputStream newInputStream(Path file) throws IOException {
        return ((MemoryMappedFileInputStream.Builder)MemoryMappedFileInputStream.builder().setPath(file)).get();
    }

    private MemoryMappedFileInputStream newInputStream(Path file, int bufferSize) throws IOException {
        return ((MemoryMappedFileInputStream.Builder)((MemoryMappedFileInputStream.Builder)MemoryMappedFileInputStream.builder().setPath(file)).setBufferSize(bufferSize)).get();
    }

    @Test
    public void testAlternateBufferSize() throws IOException {
        Path file = this.createTestFile(0x100000);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 1024);){
            Assertions.assertArrayEquals((byte[])expectedData, (byte[])IOUtils.toByteArray(inputStream));
        }
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.commons.io.input.AbstractInputStreamTest#getArrayLengths"})
    public void testAvailableAfterClose(int len) throws Exception {
        MemoryMappedFileInputStream shadow;
        Path file = this.createTestFile(len);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 1024);){
            Assertions.assertEquals((int)0, (int)((InputStream)inputStream).available());
            shadow = inputStream;
        }
        Assertions.assertEquals((int)0, (int)((InputStream)shadow).available());
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.commons.io.input.AbstractInputStreamTest#getArrayLengths"})
    public void testAvailableAfterOpen(int len) throws Exception {
        Path file = this.createTestFile(len);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 1024);){
            Assertions.assertEquals((int)0, (int)((InputStream)inputStream).available());
            ((InputStream)inputStream).read();
            Assertions.assertEquals((int)Math.max(len - 1, 0), (int)((InputStream)inputStream).available());
            IOUtils.toByteArray(inputStream);
            Assertions.assertEquals((int)0, (int)((InputStream)inputStream).available());
        }
    }

    @Test
    public void testEmptyFile() throws IOException {
        Path file = this.createTestFile(0);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file);){
            Assertions.assertArrayEquals((byte[])ArrayUtils.EMPTY_BYTE_ARRAY, (byte[])IOUtils.toByteArray(inputStream));
        }
    }

    @Test
    public void testLargerFile() throws IOException {
        Path file = this.createTestFile(0x100000);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file);){
            Assertions.assertArrayEquals((byte[])expectedData, (byte[])IOUtils.toByteArray(inputStream));
        }
    }

    @Test
    public void testReadAfterClose() throws IOException {
        Path file = this.createTestFile(0x100000);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 1024);){
            ((InputStream)inputStream).close();
            Assertions.assertThrows(IOException.class, () -> IOUtils.toByteArray(inputStream));
        }
    }

    @ParameterizedTest
    @MethodSource(value={"org.apache.commons.io.input.AbstractInputStreamTest#getArrayLengths"})
    public void testReadAfterClose(int len) throws Exception {
        Path file = this.createTestFile(len);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 1024);){
            ((InputStream)inputStream).close();
            Assertions.assertThrows(IOException.class, inputStream::read);
        }
    }

    @Test
    public void testReadSingleByte() throws IOException {
        Path file = this.createTestFile(2);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 1024);){
            int b1 = ((InputStream)inputStream).read();
            int b2 = ((InputStream)inputStream).read();
            Assertions.assertEquals((int)-1, (int)((InputStream)inputStream).read());
            Assertions.assertArrayEquals((byte[])expectedData, (byte[])new byte[]{(byte)b1, (byte)b2});
        }
    }

    @Test
    public void testSkipAtStart() throws IOException {
        Path file = this.createTestFile(100);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 10);){
            Assertions.assertEquals((long)1L, (long)((InputStream)inputStream).skip(1L));
            byte[] data = IOUtils.toByteArray(inputStream);
            Assertions.assertArrayEquals((byte[])Arrays.copyOfRange(expectedData, 1, expectedData.length), (byte[])data);
        }
    }

    @Test
    public void testSkipEmpty() throws IOException {
        Path file = this.createTestFile(0);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file);){
            Assertions.assertEquals((long)0L, (long)((InputStream)inputStream).skip(5L));
            Assertions.assertArrayEquals((byte[])ArrayUtils.EMPTY_BYTE_ARRAY, (byte[])IOUtils.toByteArray(inputStream));
        }
    }

    @Test
    public void testSkipInCurrentBuffer() throws IOException {
        Path file = this.createTestFile(100);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 10);){
            IOUtils.toByteArray((InputStream)inputStream, 5);
            Assertions.assertEquals((long)3L, (long)((InputStream)inputStream).skip(3L));
            byte[] data = IOUtils.toByteArray(inputStream);
            Assertions.assertArrayEquals((byte[])Arrays.copyOfRange(expectedData, 8, expectedData.length), (byte[])data);
        }
    }

    @ParameterizedTest
    @ValueSource(ints={-5, -1, 0})
    public void testSkipNoop(int amountToSkip) throws IOException {
        Path file = this.createTestFile(10);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file);){
            Assertions.assertEquals((long)0L, (long)((InputStream)inputStream).skip(amountToSkip));
            Assertions.assertArrayEquals((byte[])expectedData, (byte[])IOUtils.toByteArray(inputStream));
        }
    }

    @Test
    public void testSkipOutOfCurrentBuffer() throws IOException {
        Path file = this.createTestFile(100);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 10);){
            IOUtils.toByteArray((InputStream)inputStream, 5);
            Assertions.assertEquals((long)6L, (long)((InputStream)inputStream).skip(6L));
            byte[] data = IOUtils.toByteArray(inputStream);
            Assertions.assertArrayEquals((byte[])Arrays.copyOfRange(expectedData, 11, expectedData.length), (byte[])data);
        }
    }

    @Test
    public void testSkipPastEof() throws IOException {
        Path file = this.createTestFile(100);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 10);){
            IOUtils.toByteArray((InputStream)inputStream, 5);
            Assertions.assertEquals((long)95L, (long)((InputStream)inputStream).skip(96L));
            Assertions.assertArrayEquals((byte[])ArrayUtils.EMPTY_BYTE_ARRAY, (byte[])IOUtils.toByteArray(inputStream));
        }
    }

    @Test
    public void testSkipToEndOfCurrentBuffer() throws IOException {
        Path file = this.createTestFile(100);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file, 10);){
            IOUtils.toByteArray((InputStream)inputStream, 5);
            Assertions.assertEquals((long)5L, (long)((InputStream)inputStream).skip(5L));
            byte[] data = IOUtils.toByteArray(inputStream);
            Assertions.assertArrayEquals((byte[])Arrays.copyOfRange(expectedData, 10, expectedData.length), (byte[])data);
        }
    }

    @Test
    public void testSkipToEndOfCurrentBufferBuilder() throws IOException {
        Path file = this.createTestFile(100);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = ((MemoryMappedFileInputStream.Builder)((MemoryMappedFileInputStream.Builder)MemoryMappedFileInputStream.builder().setPath(file)).setBufferSize(10)).get();){
            Assertions.assertEquals((int)10, (int)inputStream.getBufferSize());
            IOUtils.toByteArray((InputStream)inputStream, 5);
            Assertions.assertEquals((long)5L, (long)inputStream.skip(5L));
            byte[] data = IOUtils.toByteArray(inputStream);
            Assertions.assertArrayEquals((byte[])Arrays.copyOfRange(expectedData, 10, expectedData.length), (byte[])data);
        }
    }

    @Test
    public void testSmallFileBuilder() throws IOException {
        Path file = this.createTestFile(100);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = ((MemoryMappedFileInputStream.Builder)MemoryMappedFileInputStream.builder().setFile(file.toFile())).get();){
            Assertions.assertArrayEquals((byte[])expectedData, (byte[])IOUtils.toByteArray(inputStream));
        }
    }

    @Test
    public void testSmallPath() throws IOException {
        Path file = this.createTestFile(100);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = this.newInputStream(file);){
            Assertions.assertArrayEquals((byte[])expectedData, (byte[])IOUtils.toByteArray(inputStream));
        }
    }

    @Test
    public void testSmallPathBuilder() throws IOException {
        Path file = this.createTestFile(100);
        byte[] expectedData = Files.readAllBytes(file);
        try (MemoryMappedFileInputStream inputStream = ((MemoryMappedFileInputStream.Builder)MemoryMappedFileInputStream.builder().setPath(file)).get();){
            Assertions.assertArrayEquals((byte[])expectedData, (byte[])IOUtils.toByteArray(inputStream));
        }
    }
}

