/*
 * Decompiled with CFR 0.152.
 */
package org.openqa.selenium.json;

import java.io.Closeable;
import java.io.IOException;
import java.io.Reader;
import java.io.UncheckedIOException;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.time.Instant;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.List;
import java.util.function.Function;
import org.jspecify.annotations.Nullable;
import org.openqa.selenium.internal.Require;
import org.openqa.selenium.json.Input;
import org.openqa.selenium.json.JsonException;
import org.openqa.selenium.json.JsonType;
import org.openqa.selenium.json.JsonTypeCoercer;
import org.openqa.selenium.json.PropertySetting;
import org.openqa.selenium.json.TypeCoercer;

public class JsonInput
implements Closeable {
    private final Reader source;
    private boolean readPerformed = false;
    private JsonTypeCoercer coercer;
    private PropertySetting setter;
    private final Input input;
    private final Deque<Container> stack = new ArrayDeque<Container>();

    JsonInput(Reader source, JsonTypeCoercer coercer, PropertySetting setter) {
        this.source = Require.nonNull("Source", source);
        this.coercer = Require.nonNull("Coercer", coercer);
        this.input = new Input(source);
        this.setter = Require.nonNull("Setter", setter);
    }

    public PropertySetting propertySetting(PropertySetting setter) {
        PropertySetting previous = this.setter;
        this.setter = Require.nonNull("Setter", setter);
        return previous;
    }

    public JsonInput addCoercers(TypeCoercer<?> ... coercers) {
        return this.addCoercers(List.of(coercers));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public JsonInput addCoercers(Iterable<TypeCoercer<?>> coercers) {
        JsonInput jsonInput = this;
        synchronized (jsonInput) {
            if (this.readPerformed) {
                throw new JsonException("JsonInput has already been used and may not be modified");
            }
            this.coercer = new JsonTypeCoercer(this.coercer, coercers);
        }
        return this;
    }

    @Override
    public void close() {
        try {
            this.source.close();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public JsonType peek() {
        this.skipWhitespace(this.input);
        switch (this.input.peek()) {
            case 'f': 
            case 't': {
                return JsonType.BOOLEAN;
            }
            case 'n': {
                return JsonType.NULL;
            }
            case '+': 
            case '-': 
            case '0': 
            case '1': 
            case '2': 
            case '3': 
            case '4': 
            case '5': 
            case '6': 
            case '7': 
            case '8': 
            case '9': {
                return JsonType.NUMBER;
            }
            case '\"': {
                return this.isReadingName() ? JsonType.NAME : JsonType.STRING;
            }
            case '{': {
                return JsonType.START_MAP;
            }
            case '}': {
                return JsonType.END_MAP;
            }
            case '[': {
                return JsonType.START_COLLECTION;
            }
            case ']': {
                return JsonType.END_COLLECTION;
            }
            case '\uffff': {
                return JsonType.END;
            }
        }
        char c = this.input.read();
        throw new JsonException("Unable to determine type from: " + c + ". " + String.valueOf(this.input));
    }

    public boolean nextBoolean() {
        this.expect(JsonType.BOOLEAN);
        return this.read(this.input.peek() == 't' ? "true" : "false", Boolean::valueOf);
    }

    public String nextName() {
        this.expect(JsonType.NAME);
        String name = this.readString();
        this.skipWhitespace(this.input);
        char read = this.input.read();
        if (read != ':') {
            throw new JsonException("Unable to read name. Expected colon separator, but saw '" + read + "'");
        }
        return name;
    }

    public @Nullable Object nextNull() {
        this.expect(JsonType.NULL);
        return this.read("null", str -> null);
    }

    public Number nextNumber() {
        char read;
        this.expect(JsonType.NUMBER);
        StringBuilder builder = new StringBuilder();
        boolean fractionalPart = false;
        while (Character.isDigit(read = this.input.peek()) || read == '+' || read == '-' || read == 'e' || read == 'E' || read == '.') {
            builder.append(this.input.read());
            fractionalPart |= read == '.';
        }
        try {
            BigDecimal number = new BigDecimal(builder.toString());
            if (fractionalPart) {
                return ((Number)number).doubleValue();
            }
            return ((Number)number).longValue();
        }
        catch (NumberFormatException e) {
            throw new JsonException("Unable to parse to a number: " + String.valueOf(builder) + ". " + String.valueOf(this.input));
        }
    }

    public String nextString() {
        this.expect(JsonType.STRING);
        return this.readString();
    }

    public @Nullable Instant nextInstant() {
        Long time = (Long)this.read((Type)((Object)Long.class));
        return null != time ? Instant.ofEpochSecond(time) : null;
    }

    public boolean hasNext() {
        if (this.stack.isEmpty()) {
            throw new JsonException("Unable to determine if an item has next when not in a container type. " + String.valueOf(this.input));
        }
        this.skipWhitespace(this.input);
        if (this.input.peek() == ',') {
            this.input.read();
            return true;
        }
        JsonType type = this.peek();
        return type != JsonType.END_COLLECTION && type != JsonType.END_MAP;
    }

    public void beginArray() {
        this.expect(JsonType.START_COLLECTION);
        this.stack.addFirst(Container.COLLECTION);
        this.input.read();
    }

    public void endArray() {
        this.expect(JsonType.END_COLLECTION);
        Container expectation = this.stack.removeFirst();
        if (expectation != Container.COLLECTION) {
            throw new JsonException("Attempt to close a JSON List, but a JSON Object was expected. " + String.valueOf(this.input));
        }
        this.input.read();
    }

    public void beginObject() {
        this.expect(JsonType.START_MAP);
        this.stack.addFirst(Container.MAP_NAME);
        this.input.read();
    }

    public void endObject() {
        this.expect(JsonType.END_MAP);
        Container expectation = this.stack.removeFirst();
        if (expectation != Container.MAP_NAME) {
            throw new JsonException("Attempt to close a JSON Map, but not ready to. " + String.valueOf(this.input));
        }
        this.input.read();
    }

    public void skipValue() {
        switch (this.peek()) {
            case BOOLEAN: {
                this.nextBoolean();
                break;
            }
            case NAME: {
                this.nextName();
                break;
            }
            case NULL: {
                this.nextNull();
                break;
            }
            case NUMBER: {
                this.nextNumber();
                break;
            }
            case START_COLLECTION: {
                this.beginArray();
                while (this.hasNext()) {
                    this.skipValue();
                }
                this.endArray();
                break;
            }
            case START_MAP: {
                this.beginObject();
                while (this.hasNext()) {
                    this.nextName();
                    this.skipValue();
                }
                this.endObject();
                break;
            }
            case STRING: {
                this.nextString();
                break;
            }
            default: {
                throw new JsonException("Cannot skip " + String.valueOf((Object)this.peek()) + ". " + String.valueOf(this.input));
            }
        }
    }

    private void markReadPerformed() {
        this.readPerformed = true;
    }

    public <T> @Nullable T read(Type type) {
        this.markReadPerformed();
        this.skipWhitespace(this.input);
        if (this.input.peek() == '\uffff') {
            return null;
        }
        return this.coercer.coerce(this, type, this.setter);
    }

    public <T> List<T> readArray(Type type) {
        ArrayList toReturn = new ArrayList();
        this.beginArray();
        while (this.hasNext()) {
            toReturn.add(this.coercer.coerce(this, type, this.setter));
        }
        this.endArray();
        return toReturn;
    }

    private boolean isReadingName() {
        return this.stack.peekFirst() == Container.MAP_NAME;
    }

    private void expect(JsonType type) {
        if (this.peek() != type) {
            throw new JsonException("Expected to read a " + String.valueOf((Object)type) + " but instead have: " + String.valueOf((Object)this.peek()) + ". " + String.valueOf(this.input));
        }
        Container top = this.stack.peekFirst();
        if (type == JsonType.NAME) {
            if (top == Container.MAP_NAME) {
                this.stack.removeFirst();
                this.stack.addFirst(Container.MAP_VALUE);
                return;
            }
            if (top != null) {
                throw new JsonException("Unexpected attempt to read name. " + String.valueOf(this.input));
            }
            return;
        }
        if (top == Container.MAP_VALUE) {
            this.stack.removeFirst();
            this.stack.addFirst(Container.MAP_NAME);
        }
    }

    private <X> X read(String toCompare, Function<String, X> mapper) {
        this.skipWhitespace(this.input);
        int toCompareLength = toCompare.length();
        for (int i = 0; i < toCompareLength; ++i) {
            char read = this.input.read();
            if (read == toCompare.charAt(i)) continue;
            throw new JsonException(String.format("Unable to read %s. Saw %s at position %d. %s", toCompare, Character.valueOf(read), i, this.input));
        }
        return mapper.apply(toCompare);
    }

    private String readString() {
        this.input.read();
        StringBuilder builder = new StringBuilder();
        block5: while (true) {
            char c = this.input.read();
            switch (c) {
                case '\uffff': {
                    throw new JsonException("Unterminated string: " + String.valueOf(builder) + ". " + String.valueOf(this.input));
                }
                case '\"': {
                    return builder.toString();
                }
                case '\\': {
                    this.readEscape(builder);
                    continue block5;
                }
            }
            builder.append(c);
        }
    }

    private void readEscape(StringBuilder builder) {
        char read = this.input.read();
        switch (read) {
            case 'b': {
                builder.append("\b");
                break;
            }
            case 'f': {
                builder.append("\f");
                break;
            }
            case 'n': {
                builder.append("\n");
                break;
            }
            case 'r': {
                builder.append("\r");
                break;
            }
            case 't': {
                builder.append("\t");
                break;
            }
            case 'u': {
                int result = 0;
                int multiplier = 4096;
                for (int i = 0; i < 4; ++i) {
                    char c = this.input.read();
                    int digit = Character.digit(c, 16);
                    if (digit == -1) {
                        throw new JsonException(c + " is not a hexadecimal digit. " + String.valueOf(this.input));
                    }
                    result += digit * multiplier;
                    multiplier /= 16;
                }
                builder.append((char)result);
                break;
            }
            case '\"': 
            case '/': 
            case '\\': {
                builder.append(read);
                break;
            }
            default: {
                throw new JsonException("Unexpected escape code: " + read + ". " + String.valueOf(this.input));
            }
        }
    }

    private void skipWhitespace(Input input) {
        while (input.peek() != '\uffff' && Character.isWhitespace(input.peek())) {
            input.read();
        }
    }

    private static enum Container {
        COLLECTION,
        MAP_NAME,
        MAP_VALUE;

    }
}

